/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.starlink.ttools.convert;

import uk.ac.starlink.pal.AngleDR;
import uk.ac.starlink.pal.Cartesian;
import uk.ac.starlink.pal.Galactic;
import uk.ac.starlink.pal.Pal;
import uk.ac.starlink.pal.Spherical;
import uk.ac.starlink.ttools.func.Times;

public abstract class SkySystem {
    public static final SkySystem ICRS = new ICRSSystem("icrs");
    public static final SkySystem FK5 = new FK5System("fk5");
    public static final SkySystem FK4 = new FK4System("fk4");
    public static final SkySystem GALACTIC = new GalSystem("galactic");
    public static final SkySystem SUPERGALACTIC = new SuperGalSystem("supergalactic");
    public static final SkySystem ECLIPTIC = new EclipticSystem("ecliptic");
    private static final SkySystem[] KNOWN_SYSTEMS = new SkySystem[]{ICRS, FK5, FK4, GALACTIC, SUPERGALACTIC, ECLIPTIC};
    private static final Pal PAL = new Pal();
    private static final double FK5_EPOCH = 2000.0;
    private static final double PI2 = 1.5707963267948966;
    private static final String EQ_UCD1 = "pos.eq.ra";
    private static final String EQ_UCD2 = "pos.eq.dec";
    private final String name_;
    private final String description_;
    private final String ucd1_;
    private final String ucd2_;
    private final String descrip1_;
    private final String descrip2_;
    private final String colname1_;
    private final String colname2_;

    protected SkySystem(String name, String description, String ucd1, String ucd2, String descrip1, String descrip2, String colname1, String colname2) {
        this.name_ = name;
        this.description_ = description;
        this.ucd1_ = ucd1;
        this.ucd2_ = ucd2;
        this.descrip1_ = descrip1;
        this.descrip2_ = descrip2;
        this.colname1_ = colname1;
        this.colname2_ = colname2;
    }

    public abstract double[] fromFK5(double var1, double var3, double var5);

    public abstract double[] toFK5(double var1, double var3, double var5);

    public String getName() {
        return this.name_;
    }

    public String getDescription() {
        return this.description_;
    }

    public String[] getCoordinateUcds() {
        return new String[]{this.ucd1_, this.ucd2_};
    }

    public String[] getCoordinateDescriptions() {
        return new String[]{this.getDescription() + " " + this.descrip1_, this.getDescription() + " " + this.descrip2_};
    }

    public String[] getCoordinateNames() {
        return new String[]{this.descrip1_, this.descrip2_};
    }

    public String[] getCoordinateColumnNames() {
        return new String[]{this.colname1_, this.colname2_};
    }

    public String toString() {
        return this.getName();
    }

    public static SkySystem[] getKnownSystems() {
        return (SkySystem[])KNOWN_SYSTEMS.clone();
    }

    public static String getSystemUsage() {
        StringBuffer sbuf = new StringBuffer("<ul>");
        for (int i = 0; i < KNOWN_SYSTEMS.length; ++i) {
            SkySystem sys = KNOWN_SYSTEMS[i];
            sbuf.append("<li><code>").append(sys.getName().toLowerCase()).append("</code>: ").append(sys.getDescription()).append(" (").append(sys.descrip1_).append(", ").append(sys.descrip2_).append(")</li>").append('\n');
        }
        sbuf.append("</ul>\n");
        return sbuf.toString();
    }

    public static SkySystem getSystemFor(String sysName) {
        String lname = sysName.toLowerCase();
        SkySystem gotSys = null;
        for (int i = 0; i < KNOWN_SYSTEMS.length; ++i) {
            SkySystem iSys = KNOWN_SYSTEMS[i];
            if (!iSys.getName().toLowerCase().startsWith(lname)) continue;
            if (gotSys == null) {
                gotSys = iSys;
                continue;
            }
            throw new IllegalArgumentException("Ambiguous system name: " + sysName);
        }
        if (gotSys == null) {
            StringBuffer sbuf = new StringBuffer().append("Unknown system name: ").append(sysName).append("\nKnown Systems: ");
            for (int i = 0; i < KNOWN_SYSTEMS.length; ++i) {
                if (i > 0) {
                    sbuf.append(", ");
                }
                sbuf.append(KNOWN_SYSTEMS[i].getName().toLowerCase());
            }
            sbuf.append('.');
            throw new IllegalArgumentException(sbuf.toString());
        }
        return gotSys;
    }

    private static boolean isLatitude(double theta) {
        return theta >= -1.5707963267948966 && theta <= 1.5707963267948966;
    }

    private static AngleDR fk5hz(AngleDR r2000, double bepoch) {
        double AS2R = 4.84813681109536E-6;
        double EPX = -9.647792254079767E-8;
        double EPY = -4.4118044980967775E-8;
        double EPZ = 1.1102233297408375E-7;
        double OMX = -1.4544410433286077E-9;
        double OMY = 2.9088820866572155E-9;
        double OMZ = 3.3936957677667517E-9;
        double[] ORTN = new double[]{-9.647792254079767E-8, -4.4118044980967775E-8, 1.1102233297408375E-7};
        double[] p5e = PAL.Dcs2c(r2000);
        double[][] r5h = PAL.Dav2m(ORTN);
        double t = 2000.0 - bepoch;
        double[] vst = new double[]{-1.4544410433286077E-9 * t, 2.9088820866572155E-9 * t, 3.3936957677667517E-9 * t};
        double[][] rst = PAL.Dav2m(vst);
        double[] p5 = PAL.Dimxv(rst, p5e);
        double[] ph = PAL.Dmxv(r5h, p5);
        AngleDR hipp = PAL.Dcc2s(ph);
        hipp.setAlpha(PAL.Dranrm(hipp.getAlpha()));
        return hipp;
    }

    private static AngleDR hfk5z(AngleDR rHipp, double bepoch) {
        double AS2R = 4.84813681109536E-6;
        double EPX = -9.647792254079767E-8;
        double EPY = -4.4118044980967775E-8;
        double EPZ = 1.1102233297408375E-7;
        double OMX = -1.4544410433286077E-9;
        double OMY = 2.9088820866572155E-9;
        double OMZ = 3.3936957677667517E-9;
        double[] ORTN = new double[]{-9.647792254079767E-8, -4.4118044980967775E-8, 1.1102233297408375E-7};
        double[] ph = PAL.Dcs2c(rHipp);
        double[][] r5h = PAL.Dav2m(ORTN);
        double[] s5 = new double[]{-1.4544410433286077E-9, 2.9088820866572155E-9, 3.3936957677667517E-9};
        double[] sh = PAL.Dmxv(r5h, s5);
        double t = bepoch - 2000.0;
        double[] vst = new double[]{-1.4544410433286077E-9 * t, 2.9088820866572155E-9 * t, 3.3936957677667517E-9 * t};
        double[][] rst = PAL.Dav2m(vst);
        double[][] r5ht = PAL.Dmxm(r5h, rst);
        double[] pv5e1 = PAL.Dimxv(r5ht, ph);
        double[] vv = PAL.Dvxv(sh, ph);
        double[] pv5e2 = PAL.Dimxv(r5ht, vv);
        Cartesian pv5e = new Cartesian(pv5e1[0], pv5e1[1], pv5e1[2], pv5e2[0], pv5e2[1], pv5e2[2]);
        Spherical sph5 = PAL.Dc62s(pv5e);
        double r5 = PAL.Dranrm(sph5.getLong());
        double d5 = sph5.getLat();
        return new AngleDR(r5, d5);
    }

    private static class EclipticSystem
    extends SkySystem {
        public EclipticSystem(String name) {
            super(name, "Ecliptic", "pos.ecliptic.lon", "pos.ecliptic.lat", "Longitude", "Latitude", "LONG", "LAT");
        }

        @Override
        public double[] fromFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                double mjd = Times.julianToMjd(epoch);
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = PAL.Eqecl(in, mjd);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }

        @Override
        public double[] toFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                double mjd = Times.julianToMjd(epoch);
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = PAL.Ecleq(in, mjd);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }
    }

    private static class SuperGalSystem
    extends SkySystem {
        public SuperGalSystem(String name) {
            super(name, "de Vaucouleurs Supergalactic", "pos.supergalactic.lon", "pos.supergalactic.lat", "Longitude", "Latitude", "SUPERGAL_LONG", "SUPERGAL_LAT");
        }

        @Override
        public double[] fromFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                Galactic out = PAL.Galsup(PAL.Eqgal(in));
                return new double[]{out.getLongitude(), out.getLatitude()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }

        @Override
        public double[] toFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                Galactic in = new Galactic(c1, c2);
                AngleDR out = PAL.Galeq(PAL.Supgal(in));
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }
    }

    private static class GalSystem
    extends SkySystem {
        public GalSystem(String name) {
            super(name, "IAU 1958 Galactic", "pos.galactic.lon", "pos.galactic.lat", "Longitude", "Latitude", "GAL_LONG", "GAL_LAT");
        }

        @Override
        public double[] fromFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                Galactic out = PAL.Eqgal(in);
                return new double[]{out.getLongitude(), out.getLatitude()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }

        @Override
        public double[] toFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                Galactic in = new Galactic(c1, c2);
                AngleDR out = PAL.Galeq(in);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }
    }

    private static class FK4System
    extends SkySystem {
        public FK4System(String name) {
            super(name, "FK4 B1950.0", SkySystem.EQ_UCD1, SkySystem.EQ_UCD2, "Right Ascension", "Declination", "RA1950", "DEC1950");
        }

        @Override
        public double[] fromFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = PAL.Fk54z(in, epoch).getAngle();
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }

        @Override
        public double[] toFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = PAL.Fk45z(in, epoch);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }
    }

    private static class ICRSSystem
    extends SkySystem {
        public ICRSSystem(String name) {
            super(name, "ICRS", SkySystem.EQ_UCD1, SkySystem.EQ_UCD2, "Right Ascension", "Declination", "RA", "DEC");
        }

        @Override
        public double[] fromFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = SkySystem.fk5hz(in, epoch);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }

        @Override
        public double[] toFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = SkySystem.hfk5z(in, epoch);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }
    }

    private static class FK5System
    extends SkySystem {
        public FK5System(String name) {
            super(name, "FK5 J2000.0", SkySystem.EQ_UCD1, SkySystem.EQ_UCD2, "Right Ascension", "Declination", "RA2000", "DEC2000");
        }

        @Override
        public double[] fromFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = PAL.Preces("FK5", epoch, 2000.0, in);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }

        @Override
        public double[] toFK5(double c1, double c2, double epoch) {
            if (SkySystem.isLatitude(c2)) {
                AngleDR in = new AngleDR(c1, c2);
                AngleDR out = PAL.Preces("FK5", 2000.0, epoch, in);
                return new double[]{out.getAlpha(), out.getDelta()};
            }
            return new double[]{Double.NaN, Double.NaN};
        }
    }
}

