/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.starlink.ttools.plot2;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.function.DoubleUnaryOperator;
import java.util.logging.Level;
import java.util.logging.Logger;
import uk.ac.starlink.ttools.func.Maths;
import uk.ac.starlink.ttools.plot2.BasicRanger;
import uk.ac.starlink.ttools.plot2.HistoRanger;
import uk.ac.starlink.ttools.plot2.PlotUtil;
import uk.ac.starlink.ttools.plot2.Ranger;
import uk.ac.starlink.ttools.plot2.Scaler;
import uk.ac.starlink.ttools.plot2.Scaling;
import uk.ac.starlink.ttools.plot2.Span;

public class Scalings {
    private static final Object SQRT_TYPE = new Object();
    private static final Object SQR_TYPE = new Object();
    private static final Object ACOS_TYPE = new Object();
    private static final Object COS_TYPE = new Object();
    private static final double AUTO_DELTA = 0.0625;
    private static final double UNSCALE_TOL = 1.0E-4;
    private static final double UNSCALE_MAXIT = 50.0;
    private static final Scaling.RangeScaling LINEAR = Scalings.createLinearScaling("Linear");
    private static final Logger logger_ = Logger.getLogger("uk.ac.starlink.ttools.plot2.layer");

    private Scalings() {
    }

    public static Ranger createRanger(Scaling[] scalings) {
        boolean hasRange = false;
        boolean hasHisto = false;
        boolean hasOther = false;
        for (Scaling scaling : scalings) {
            if (scaling instanceof Scaling.RangeScaling) {
                hasRange = true;
                continue;
            }
            if (scaling instanceof Scaling.HistogramScaling) {
                hasHisto = true;
                continue;
            }
            if (scaling == null) continue;
            logger_.warning("Unknown scaling type: " + scaling);
            hasOther = true;
        }
        if (hasOther) {
            return new BasicRanger(true);
        }
        if (hasHisto) {
            return new HistoRanger(100000, 1000);
        }
        return new BasicRanger(false);
    }

    public static boolean canScale(Scaling[] scalings, Span dataSpan, Span fixSpan) {
        boolean hasRange = false;
        boolean hasHisto = false;
        boolean hasOther = false;
        for (Scaling scaling : scalings) {
            if (scaling instanceof Scaling.RangeScaling) {
                hasRange = true;
                continue;
            }
            if (scaling instanceof Scaling.HistogramScaling) {
                hasHisto = true;
                continue;
            }
            if (scaling == null) continue;
            assert (false);
            logger_.warning("Unknown scaling type: " + scaling);
            hasOther = true;
        }
        if (hasOther) {
            return false;
        }
        if (!(!hasHisto || Scalings.isFiniteSpan(dataSpan) && HistoRanger.canScaleHistograms(dataSpan))) {
            return false;
        }
        return Scalings.isFiniteSpan(dataSpan) || Scalings.isFiniteSpan(fixSpan);
    }

    static Scaling.RangeScaling createLinearScaling(String name) {
        return new ClippedScaling(name, "Linear scaling", false){
            final Scaling type = this;

            @Override
            public Scaler createClippedScaler(final double lo, double hi) {
                final double scale = 1.0 / (hi - lo);
                return new DefaultScaler(false, lo, hi, this.type){

                    @Override
                    public double scaleValue(double val) {
                        return (val - lo) * scale;
                    }
                };
            }
        };
    }

    static Scaling.RangeScaling createLogScaling(String name) {
        return new ClippedScaling(name, "Logarithmic scaling, positive values only", true){
            final Scaling type = this;

            @Override
            public Scaler createClippedScaler(double lo, double hi) {
                double xlo;
                if (lo > 0.0) {
                    xlo = lo;
                } else if (hi > 1.0) {
                    xlo = 1.0;
                } else if (hi > 0.0) {
                    xlo = hi * 0.001;
                } else {
                    xlo = 0.1;
                    hi = 10.0;
                }
                final double base1 = 1.0 / xlo;
                final double scale = 1.0 / (Math.log(hi) - Math.log(xlo));
                return new DefaultScaler(true, xlo, hi, this.type){

                    @Override
                    public double scaleValue(double val) {
                        return val > 0.0 ? Math.log(val * base1) * scale : 0.0;
                    }
                };
            }
        };
    }

    static Scaling.RangeScaling createAutoScaling(final String name) {
        final Scaling.RangeScaling asinh = Scalings.createAutoScaling("Asinh-auto", 0.0625);
        double minSpan = 17.0;
        String descrip = "asinh-based scaling with default parameters";
        return new Scaling.RangeScaling(){

            @Override
            public String getName() {
                return name;
            }

            @Override
            public String getDescription() {
                return "asinh-based scaling with default parameters";
            }

            @Override
            public boolean isLogLike() {
                return false;
            }

            @Override
            public Scaler createScaler(double lo, double hi) {
                return asinh.createScaler(lo, Math.max(hi, lo + 17.0));
            }
        };
    }

    static Scaling.RangeScaling createSqrtScaling(String name) {
        return new ReScaling(name, "Square root scaling", LINEAR, new DefaultScaler(true, 0.0, 1.0, SQRT_TYPE){

            @Override
            public double scaleValue(double val) {
                return Math.sqrt(val);
            }
        });
    }

    static Scaling.RangeScaling createAsinhScaling(String name) {
        String descrip = "Asinh scaling, wide dynamic range for both positive and negative values";
        return new ClippedScaling(name, descrip, false){
            final Scaling type = this;

            @Override
            public Scaler createClippedScaler(double lo, double hi) {
                double amax = Math.max(Math.abs(lo), Math.abs(hi));
                double a1 = Math.sinh(4.0) / amax;
                final DoubleUnaryOperator f = d -> Maths.asinh(a1 * d);
                final double min = f.applyAsDouble(lo);
                double max = f.applyAsDouble(hi);
                final double range1 = 1.0 / (max - min);
                return new DefaultScaler(false, lo, hi, this.type){

                    @Override
                    public double scaleValue(double val) {
                        return (f.applyAsDouble(val) - min) * range1;
                    }
                };
            }
        };
    }

    static Scaling.RangeScaling createSquareScaling(String name) {
        return new ReScaling(name, "Square scaling", LINEAR, new DefaultScaler(true, 0.0, 1.0, SQR_TYPE){

            @Override
            public double scaleValue(double val) {
                return val * val;
            }
        });
    }

    static Scaling.RangeScaling createAcosScaling(String name) {
        return new ReScaling(name, "Inverse cosine Scaling", LINEAR, new DefaultScaler(false, 0.0, 1.0, ACOS_TYPE){

            @Override
            public double scaleValue(double val) {
                return Math.acos(1.0 - 2.0 * val) / Math.PI;
            }
        });
    }

    static Scaling.RangeScaling createCosScaling(String name) {
        return new ReScaling(name, "Cosine Scaling", LINEAR, new DefaultScaler(false, 0.0, 1.0, COS_TYPE){

            @Override
            public double scaleValue(double val) {
                return 0.5 * (1.0 + Math.cos((1.0 + val) * Math.PI));
            }
        });
    }

    static Scaling.HistogramScaling createHistogramScaling(final String name, final boolean isLogLike) {
        final String descrip = "Scaling follows data distribution, with " + (isLogLike ? "logarithmic" : "linear") + " axis" + (isLogLike ? ", positive values only" : "");
        return new Scaling.HistogramScaling(){

            @Override
            public String getName() {
                return name;
            }

            @Override
            public String getDescription() {
                return descrip;
            }

            @Override
            public boolean isLogLike() {
                return isLogLike;
            }

            public String toString() {
                return name;
            }
        };
    }

    public static Scaling.RangeScaling createAutoScaling(String name, double delta) {
        String descrip = "asinh-based scaling in which a unit difference at the bottom of the input scale translates to a difference of " + delta + " in the output";
        return new AsinhScaling(name, descrip, delta);
    }

    public static double unscale(Scaler scaler, double lo, double hi, double frac) {
        return Scalings.unscaleBisect(scaler, lo, hi, frac);
    }

    private static double unscaleBisect(Scaler scaler, double lo, double hi, double frac) {
        boolean n = false;
        double dLo = lo;
        double dHi = hi;
        double fTarget = frac;
        double fLo = scaler.scaleValue(dLo);
        double fHi = scaler.scaleValue(dHi);
        if (fLo == frac) {
            return dLo;
        }
        if (fHi == frac) {
            return dHi;
        }
        int i = 0;
        while ((double)i < 50.0) {
            double dMid = 0.5 * (dLo + dHi);
            double fMid = scaler.scaleValue(dMid);
            assert (fMid >= 0.0 && fMid <= 1.0);
            assert (dMid >= lo && dMid <= hi);
            if (Math.abs(fMid - fTarget) < 1.0E-4) {
                return dMid;
            }
            if ((fLo - fTarget) / (fMid - fTarget) > 0.0) {
                fLo = fMid;
                dLo = dMid;
            } else {
                fHi = fMid;
                dHi = dMid;
            }
            ++i;
        }
        Level level = Level.INFO;
        if (logger_.isLoggable(level)) {
            logger_.info("Unscale did not converge after 50.0 iterations");
        }
        return lo + frac * (hi - lo);
    }

    private static double unscaleSecant(Scaler scaler, double lo, double hi, double frac) {
        double fTarget = frac;
        double d1 = lo;
        double d2 = hi;
        double f1 = scaler.scaleValue(d1);
        double f2 = scaler.scaleValue(d2);
        int i = 0;
        while ((double)i < 50.0) {
            double d0 = (d2 * (f1 - fTarget) - d1 * (f2 - fTarget)) / (f1 - fTarget - (f2 - fTarget));
            double f0 = scaler.scaleValue(d0);
            if (Math.abs(f0 - fTarget) < 1.0E-4) {
                return d0;
            }
            d2 = d1;
            f2 = f1;
            d1 = d0;
            f1 = f0;
            ++i;
        }
        Level level = Level.INFO;
        if (logger_.isLoggable(level)) {
            logger_.info("Unscale did not converge after 50.0 iterations");
        }
        return lo + frac * (hi - lo);
    }

    private static boolean isFiniteSpan(Span span) {
        if (span != null) {
            return PlotUtil.isFinite(span.getLow()) && PlotUtil.isFinite(span.getHigh());
        }
        return false;
    }

    private static class AsinhScale {
        private final double u_;
        private final double v_;

        AsinhScale(double delta, double max) {
            this(AsinhScale.calcCoeffs(delta, max));
            assert (this.scaleValue(0.0) == 0.0);
            assert (PlotUtil.approxEquals(delta, this.scaleValue(1.0)));
            assert (this.scaleValue(max) <= 1.0);
        }

        private AsinhScale(double[] coeffs) {
            this.u_ = coeffs[0];
            this.v_ = coeffs[1];
        }

        public double scaleValue(double c) {
            return this.u_ * Maths.asinh(this.v_ * c);
        }

        private static double[] calcCoeffs(double delta, double max) {
            double v0 = 1.0;
            boolean done = false;
            while (!done) {
                double[] derivs = AsinhScale.calcDerivsV(v0, max, delta);
                double v1 = v0 - derivs[0] / derivs[1];
                done = Math.abs(v1 - v0) < 1.0E-14;
                v0 = v1;
            }
            double v = v0;
            double u = 1.0 / Maths.asinh(v * max);
            return new double[]{u, v};
        }

        private static double[] calcDerivsV(double v, double max, double delta) {
            double d0 = Maths.sinh(delta * Maths.asinh(v * max)) - v;
            double d1 = Maths.cosh(delta * Maths.asinh(v * max)) * delta * max / Math.hypot(v * max, 1.0) - 1.0;
            return new double[]{d0, d1};
        }
    }

    private static abstract class DefaultScaler
    implements Scaler {
        private final boolean isLogLike_;
        private final double lo_;
        private final double hi_;
        private final Object type_;

        DefaultScaler(boolean isLogLike, double lo, double hi, Object type) {
            this.isLogLike_ = isLogLike;
            this.lo_ = lo;
            this.hi_ = hi;
            this.type_ = type;
        }

        @Override
        public boolean isLogLike() {
            return this.isLogLike_;
        }

        @Override
        public double getLow() {
            return this.lo_;
        }

        @Override
        public double getHigh() {
            return this.hi_;
        }

        public int hashCode() {
            int code = 990;
            code = 23 * code + Float.floatToIntBits((float)this.lo_);
            code = 23 * code + Float.floatToIntBits((float)this.hi_);
            code = 23 * code + this.type_.hashCode();
            return code;
        }

        public boolean equals(Object o) {
            if (o instanceof DefaultScaler) {
                DefaultScaler other = (DefaultScaler)o;
                return this.lo_ == other.lo_ && this.hi_ == other.hi_ && this.type_.equals(other.type_);
            }
            return false;
        }
    }

    private static abstract class ClippedScaling
    extends DefaultScaling
    implements Scaling.RangeScaling {
        protected ClippedScaling(String name, String description, boolean isLogLike) {
            super(name, description, isLogLike);
        }

        protected abstract Scaler createClippedScaler(double var1, double var3);

        @Override
        public Scaler createScaler(final double lo, final double hi) {
            if (lo < hi) {
                final Scaler clipScaler = this.createClippedScaler(lo, hi);
                final double loOut = clipScaler.scaleValue(lo);
                final double hiOut = clipScaler.scaleValue(hi);
                return new DefaultScaler(this.isLogLike(), lo, hi, clipScaler){

                    @Override
                    public double scaleValue(double val) {
                        if (val <= lo) {
                            return loOut;
                        }
                        if (val >= hi) {
                            return hiOut;
                        }
                        if (Double.isNaN(val)) {
                            return Double.NaN;
                        }
                        return clipScaler.scaleValue(val);
                    }
                };
            }
            if (lo == hi) {
                final double midVal = lo;
                double elo = midVal * 0.999;
                double ehi = midVal * 1.001;
                Scaler clipScaler = this.createClippedScaler(elo, ehi);
                final double loOut = ClippedScaling.clipUnit(clipScaler.scaleValue(elo));
                final double hiOut = ClippedScaling.clipUnit(clipScaler.scaleValue(ehi));
                double midOut = clipScaler.scaleValue(midVal);
                return new DefaultScaler(this.isLogLike(), elo, ehi, clipScaler){

                    @Override
                    public double scaleValue(double val) {
                        if (val < midVal) {
                            return loOut;
                        }
                        if (val > midVal) {
                            return hiOut;
                        }
                        if (val == midVal) {
                            return midVal;
                        }
                        return Double.NaN;
                    }
                };
            }
            throw new IllegalArgumentException("! " + lo + " < " + hi);
        }

        private static double clipUnit(double val) {
            return Math.min(1.0, Math.max(0.0, val));
        }
    }

    private static class ReScaling
    extends DefaultScaling
    implements Scaling.RangeScaling {
        private final Scaling.RangeScaling baseScaling_;
        private final Scaler rescaler_;

        public ReScaling(String name, String description, Scaling.RangeScaling baseScaling, Scaler rescaler) {
            super(name, description, baseScaling.isLogLike());
            this.baseScaling_ = baseScaling;
            this.rescaler_ = rescaler;
        }

        @Override
        public Scaler createScaler(double lo, double hi) {
            final Scaler baseScaler = this.baseScaling_.createScaler(lo, hi);
            ArrayList<Scaler> type = new ArrayList<Scaler>(Arrays.asList(this.rescaler_, baseScaler));
            return new DefaultScaler(this.baseScaling_.isLogLike(), lo, hi, type){

                @Override
                public double scaleValue(double val) {
                    return rescaler_.scaleValue(baseScaler.scaleValue(val));
                }
            };
        }

        public int hashCode() {
            int code = 33441;
            code = 23 * code + this.baseScaling_.hashCode();
            code = 23 * code + this.rescaler_.hashCode();
            return code;
        }

        public boolean equals(Object o) {
            if (o instanceof ReScaling) {
                ReScaling other = (ReScaling)o;
                return this.baseScaling_ == other.baseScaling_ && this.rescaler_ == other.rescaler_;
            }
            return false;
        }
    }

    private static class AsinhScaling
    extends ClippedScaling {
        private final double delta_;

        AsinhScaling(String name, String description, double delta) {
            super(name, description, false);
            this.delta_ = delta;
        }

        @Override
        public Scaler createClippedScaler(final double lo, double hi) {
            final AsinhScale zScaler = new AsinhScale(this.delta_, hi - lo);
            return new DefaultScaler(false, lo, hi, this){

                @Override
                public double scaleValue(double val) {
                    return zScaler.scaleValue(val - lo);
                }
            };
        }

        public int hashCode() {
            int code = 79982;
            code = 23 * code + Float.floatToIntBits((float)this.delta_);
            return code;
        }

        public boolean equals(Object o) {
            return o instanceof AsinhScaling && this.delta_ == ((AsinhScaling)o).delta_;
        }
    }

    private static abstract class DefaultScaling
    implements Scaling {
        final String name_;
        final String description_;
        final boolean isLogLike_;

        DefaultScaling(String name, String description, boolean isLogLike) {
            this.name_ = name;
            this.description_ = description;
            this.isLogLike_ = isLogLike;
        }

        @Override
        public String getName() {
            return this.name_;
        }

        @Override
        public String getDescription() {
            return this.description_;
        }

        @Override
        public boolean isLogLike() {
            return this.isLogLike_;
        }

        public String toString() {
            return this.getName();
        }
    }
}

