/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.starlink.ttools.plot2.data;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Formatter;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class DiskCache {
    private final File dir_;
    private final Map<File, Long> createdFiles_;
    private final Level logLevel_;
    private final Logger logger_;
    private long limit_;
    public static final String CACHE_DIRNAME = "stilts-cache";
    public static final String README_NAME = "README-cache.txt";
    private static final String UTF8 = "UTF-8";

    public DiskCache(File dir, long limit) {
        this.dir_ = dir;
        this.limit_ = limit;
        this.logger_ = Logger.getLogger("uk.ac.starlink.ttools.plot2.data");
        this.logLevel_ = Level.WARNING;
        this.createdFiles_ = new ConcurrentHashMap<File, Long>();
        Runtime.getRuntime().addShutdownHook(new Thread("Cache shutdown logging"){

            @Override
            public void run() {
                if (DiskCache.this.createdFiles_.size() > 0) {
                    System.err.println("NOTE: " + DiskCache.this.getWriteSummary());
                }
            }
        });
    }

    public File getDir() {
        return this.dir_;
    }

    public void fileAdded(File f) {
        this.createdFiles_.put(f, f.length());
    }

    public void ready() throws IOException {
        if (DiskCache.mkdirs(this.dir_, false)) {
            this.logger_.log(this.logLevel_, "Created plot cache directory " + this.dir_);
            this.initBaseCacheDir(DiskCache.getNamedAncestor(this.dir_, CACHE_DIRNAME));
        }
        if (!this.dir_.isDirectory() || !this.dir_.canWrite()) {
            throw new IOException("Cache path " + this.dir_ + " is not a writable directory");
        }
    }

    public void log(String txt) {
        this.logger_.log(this.logLevel_, txt);
    }

    public void tidy() {
        if (this.limit_ == 0L) {
            this.limit_ = DiskCache.getDefaultCacheLimit(this.dir_);
        }
        long freeSpace = this.dir_.getUsableSpace();
        if (this.limit_ < 0L && freeSpace > -this.limit_) {
            this.logger_.log(this.logLevel_, "Plot cache filesystem free space " + DiskCache.formatByteSize(freeSpace) + " > " + DiskCache.formatByteSize(-this.limit_));
            return;
        }
        File[] files = this.dir_.listFiles();
        ArrayList<FileMeta> fms = new ArrayList<FileMeta>();
        long totsize = 0L;
        for (File f : files) {
            FileMeta fm = new FileMeta(f);
            fms.add(fm);
            totsize += fm.size_;
        }
        long maxSize = this.limit_ > 0L ? this.limit_ : freeSpace + totsize + this.limit_;
        this.log("Plot cache usage: " + DiskCache.formatByteSize(totsize) + " / " + DiskCache.formatByteSize(maxSize));
        long overSize = totsize - maxSize;
        if (overSize > 0L) {
            Collections.sort(fms, FileMeta.BY_LAST_MODIFIED);
            for (FileMeta fm : fms) {
                if (overSize <= 0L) {
                    return;
                }
                File file = fm.file_;
                long size = fm.size_;
                if (file.delete()) {
                    overSize -= size;
                    this.log("Drop plot cache file " + file + " (" + DiskCache.formatByteSize(size) + ")");
                    continue;
                }
                this.logger_.warning("Failed to drop plot cache file " + file);
            }
        }
    }

    public String getWriteSummary() {
        long nbyte = 0L;
        for (Long fsize : this.createdFiles_.values()) {
            nbyte += fsize.longValue();
        }
        int nfile = this.createdFiles_.size();
        return new StringBuffer().append("Persistent plot cache data written: ").append(nfile).append(" ").append(nfile == 1 ? "file" : "files").append(", ").append(DiskCache.formatByteSize(nbyte)).append(" in directory ").append(this.dir_).toString();
    }

    public void touch(File file) {
        if (!file.setLastModified(System.currentTimeMillis())) {
            this.logger_.warning("Touch " + file + " failed");
        }
    }

    private void initBaseCacheDir(File dir) {
        if (dir == null) {
            return;
        }
        File readmeFile = new File(dir, README_NAME);
        if (readmeFile.exists()) {
            return;
        }
        try {
            PrintStream out = new PrintStream(new FileOutputStream(readmeFile));
            out.println("This directory holds persistent cached data");
            out.println("for use by the STILTS application.");
            out.println("It's probably related to plotting operations.");
            out.println("Deleting this directory or its contents");
            out.println("is not likely to break anything,");
            out.println("but it may lead to slower startup of some plots");
            out.println("next time they are invoked.");
            out.close();
        }
        catch (IOException e) {
            this.logger_.log(Level.WARNING, "Failed to write " + readmeFile, e);
        }
    }

    public static String hashText(String txt) {
        return DiskCache.hashText(txt, 16);
    }

    public static String hashText(String txt, int nchar) {
        byte[] txtBytes;
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            assert (false);
            return txt.substring(0, Math.min(nchar, txt.length()));
        }
        try {
            txtBytes = txt.getBytes(UTF8);
        }
        catch (UnsupportedEncodingException e) {
            assert (false);
            txtBytes = txt.getBytes();
        }
        byte[] mdbytes = md.digest(txtBytes);
        int nbyte = Math.min(nchar / 2, mdbytes.length);
        Formatter formatter = new Formatter();
        for (int i = 0; i < nbyte; ++i) {
            formatter.format("%02x", mdbytes[i]);
        }
        return formatter.toString();
    }

    public static File toWorkFilename(File file) {
        String parent = file.getParent();
        String name = file.getName();
        int dotIx = name.lastIndexOf(46);
        String workId = "W-" + String.format("%08x", (int)System.nanoTime());
        String rename = dotIx >= 0 ? name.substring(0, dotIx) + "-" + workId + name.substring(dotIx) : workId + "-" + name;
        return new File(parent, rename);
    }

    public static String formatByteSize(long nbyte) {
        long kfact = 1000L;
        long kilo = kfact * 1L;
        long mega = kfact * kilo;
        long giga = kfact * mega;
        long tera = kfact * giga;
        long absNbyte = Math.abs(nbyte);
        if (absNbyte >= tera * 10L) {
            return nbyte / tera + "TB";
        }
        if (absNbyte >= giga * 10L) {
            return nbyte / giga + "GB";
        }
        if (absNbyte >= mega * 10L) {
            return nbyte / mega + "MB";
        }
        if (absNbyte >= kilo * 10L) {
            return nbyte / kilo + "kB";
        }
        return nbyte + "bytes";
    }

    public static File getSystemTmpDir() {
        return new File(System.getProperty("java.io.tmpdir"));
    }

    public static File toCacheDir(File baseDir, String label) {
        if (baseDir == null) {
            baseDir = new File(System.getProperty("java.io.tmpdir"));
        }
        return new File(new File(baseDir, CACHE_DIRNAME), label);
    }

    public static boolean mkdirs(File dir, boolean ownerOnly) {
        return dir.exists() ? false : DiskCache.createDirIfNecessary(dir, ownerOnly);
    }

    private static boolean createDirIfNecessary(File dir, boolean ownerOnly) {
        if (dir.exists()) {
            return true;
        }
        if (DiskCache.createDirIfNecessary(dir.getParentFile(), ownerOnly)) {
            if (dir.mkdir()) {
                dir.setWritable(true, ownerOnly);
                return true;
            }
            return false;
        }
        return false;
    }

    public static long getDefaultCacheLimit(File dir) {
        long cacheSize = 0L;
        for (File f : dir.listFiles()) {
            cacheSize += f.length();
        }
        long freeSize = dir.getUsableSpace();
        long availSize = freeSize + cacheSize;
        return -availSize / 2L;
    }

    private static File getNamedAncestor(File file, String name) {
        if (file == null) {
            return null;
        }
        if (name.equals(file.getName())) {
            return file;
        }
        return DiskCache.getNamedAncestor(file.getParentFile(), name);
    }

    private static class FileMeta {
        final File file_;
        final long size_;
        final long lastModified_;
        static final Comparator<FileMeta> BY_LAST_MODIFIED = (fm1, fm2) -> Long.compare(fm1.lastModified_, fm2.lastModified_);

        FileMeta(File file) {
            this.file_ = file;
            this.size_ = file.length();
            this.lastModified_ = file.lastModified();
        }

        boolean isBad() {
            return this.size_ <= 0L || this.lastModified_ <= 0L;
        }
    }
}

