package fit.formats;

import fit.framework.MetadataItem;
import fit.framework.ObservationSet;
import fit.util.FitUtils;
import fit.util.MetaStarTable;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.logging.Logger;
import uk.ac.starlink.table.ColumnData;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.ColumnStarTable;
import uk.ac.starlink.table.DefaultValueInfo;
import uk.ac.starlink.table.DescribedValue;
import uk.ac.starlink.table.ObjectArrayColumn;
import uk.ac.starlink.table.PrimitiveArrayColumn;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StarTableWriter;
import uk.ac.starlink.table.ValueInfo;
import uk.ac.starlink.votable.VOTableWriter;

/**
 * Serialises an ObservationSet to a VOTable.
 *
 * @author   Mark Taylor
 * @since    1 Feb 2007
 */
public class ObservationSetWriter {

    private final StarTableWriter tWriter_ = new VOTableWriter();
    private final Logger logger_ = Logger.getLogger( "fit.formats" );

    /** Metadata for Y values. */
    final static DefaultValueInfo Y_INFO = 
        new DefaultValueInfo( "Y", double[].class, "Y values" );

    /** Metadata for Y errors. */
    final static DefaultValueInfo YERR_INFO =
        new DefaultValueInfo( "Y_ERR", double[].class, "Y errors" );

    /** Metadata for X shift values. */
    final static DefaultValueInfo SHIFT_INFO =
        new DefaultValueInfo( "SHIFT", Double.class,
                              "X shift (like redshift)" );

    /** Metadata for X values. */
    final static DefaultValueInfo X_INFO =
        new DefaultValueInfo( "X", double[].class, "X values" );

    /** Metadata for X errors. */
    final static DefaultValueInfo XERR_INFO =
        new DefaultValueInfo( "X_ERR", double[].class, "X errors" );

    static {
        Y_INFO.setUCD( "phot.flux" );
        YERR_INFO.setUCD( "stat.err;phot.flux" );
        X_INFO.setUCD( "em.wl.central" );
        XERR_INFO.setUCD( "stat.err;em.wl.central" );
        SHIFT_INFO.setUCD( "src.redshift.phot" );
    }

    /**
     * Serializes an observation set to an output stream.
     * Uses the stream raw and does not close it.
     *
     * @param   obsSet  observation set
     * @param   out  destination stream
     */
    public void write( ObservationSet obsSet, OutputStream out )
            throws IOException {
        tWriter_.writeStarTable( toStarTable( obsSet ), out );
    }

    /**
     * Converts an ObservationSet to a table.
     * Note the returned table has significant amounts of metadata - currently
     * the only table output handler which can honour all the important bits
     * is the VOTable one.
     *
     * @param  obsSet  observation set
     * @return  tabular equivalent
     */
    private StarTable toStarTable( ObservationSet obsSet ) {
        int nobs = obsSet.getObsCount();
        ColumnStarTable table = ColumnStarTable.makeTableWithRows( nobs );
        List params = table.getParameters();

        /* Add parameters for the X and X error vectors. */
        MetadataItem metaX = obsSet.getMetadataX();
        String xUnit = metaX.getUnit();
        String xName = metaX.getName();
        DefaultValueInfo xInfo = new DefaultValueInfo( X_INFO );
        DefaultValueInfo xerrInfo = new DefaultValueInfo( XERR_INFO );
        if ( xUnit != null && xUnit.trim().length() > 0 ) {
            xInfo.setUnitString( xUnit );
            xerrInfo.setUnitString( xUnit );
        }
        if ( xName != null && xName.trim().length() > 0 ) {
            xInfo.setName( xName );
            xerrInfo.setName( xName + "_err" );
        }
        double[] xs = FitUtils.getXs( obsSet );
        double[] xerrs = FitUtils.getXErrors( obsSet );
        params.add( new DescribedValue( xInfo, xs ) );
        params.add( new DescribedValue( xerrInfo, xerrs ) );

        /* Add columns for Y and Y error vectors. */
        MetadataItem metaY = obsSet.getMetadataY();
        String yUnit = metaY.getUnit();
        String yName = metaY.getName();
        ColumnInfo yInfo = new ColumnInfo( Y_INFO );
        ColumnInfo yerrInfo = new ColumnInfo( YERR_INFO );
        if ( yUnit != null && yUnit.trim().length() > 0 ) {
            yInfo.setUnitString( yUnit );
            yerrInfo.setUnitString( yUnit );
        }
        if ( yName != null && yName.trim().length() > 0 ) {
            yInfo.setName( yName );
            yerrInfo.setName( yName + "_err" );
        }
        double[][] ys = new double[ nobs ][];
        double[][] yerrs = new double[ nobs ][];
        for ( int iobs = 0; iobs < nobs; iobs++ ) {
            ys[ iobs ] = FitUtils.getYs( obsSet, iobs );
            yerrs[ iobs ] = FitUtils.getYErrors( obsSet, iobs );
        }
        table.addColumn( new ObjectArrayColumn( yInfo, ys ) );
        table.addColumn( new ObjectArrayColumn( yerrInfo, yerrs ) );

        /* Add a column for the (red)shift, if applicable. */
        boolean useShift = false;
        double[] factors = new double[ nobs ];
        for ( int iobs = 0; iobs < nobs; iobs++ ) {
            double factor = obsSet.getXFactor( iobs );
            if ( factor <= 0.0 ) {
                factor = 1.0;
                logger_.warning( "Ignoring nonsensical shift " + factor );
            }
            else if ( factor != 1.0 ) {
                useShift = true;
            }
            factors[ iobs ] = factor;
        }
        if ( useShift ) {
            ColumnInfo info = new ColumnInfo( SHIFT_INFO );
            table.addColumn( PrimitiveArrayColumn
                            .makePrimitiveColumn( info, factors ) );
        }

        /* Add remaining columns from metadata table. */
        if ( obsSet.getMetadataTable() != null ) {
            final StarTable metaTable =
                new MetaStarTable( obsSet.getMetadataTable() );
            int ncol = metaTable.getColumnCount();
            for ( int icol = 0; icol < ncol; icol++ ) {
                final int ic = icol;
                ValueInfo info =
                    new DefaultValueInfo( metaTable.getColumnInfo( icol ) );
                table.addColumn( new ColumnData( info ) {
                    public Object readValue( long irow ) throws IOException {
                        return metaTable.getCell( irow, ic );
                    }
                } );
            }
        }
        return table;
    }
}
