package fit.gui;

import fit.framework.Comparison;
import fit.framework.MetadataItem;
import fit.framework.ObservationSet;
import fit.framework.TheorySet;
import fit.util.FitUtils;
import java.awt.BorderLayout;
import java.awt.Color;
import java.util.logging.Logger;
import javax.swing.JPanel;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.data.Range;
import org.jfree.data.general.DatasetUtilities;

/**
 * Component which displays an observation and zero or more theoretical 
 * fits to it.
 *
 * @author   Mark Taylor
 * @since    1 Nov 2006
 */
public class ResultPlot extends JPanel {

    private final ObservationSet obsSet_;
    private final ObservationSetDataset obsDataset_;
    private final ComparisonDataset compDataset_;
    private final XYPlot plot_;
    private final Logger logger_ = Logger.getLogger( "fit.gui" );
    private boolean mismatchWarnX_;
    private boolean mismatchWarnY_;

    /**
     * Constructor.
     *
     * @param  obsSet  observation set
     */
    public ResultPlot( ObservationSet obsSet ) {
        super( new BorderLayout() );
        obsSet_ = obsSet;

        /* Create the dataset which will provide the data to plot. */
        obsDataset_ = new ObservationSetDataset( obsSet );
        obsDataset_.setSelected( new boolean[ obsSet.getObsCount() ] );
        compDataset_ = new ComparisonDataset( obsSet );

        /* Set up axes. */
        final NumericAxis xAxis =
            new NumericAxis( obsSet.getMetadataX().toString(), false,
                             obsSet.getX( 0 ) );
        final NumericAxis yAxis =
            new NumericAxis( obsSet.getMetadataY().toString(), false,
                             obsSet.getY( 0, 0 ) );

        /* Create the plot.  Arrange that default range determination is
         * done by a plot which considers only the range of the observation
         * dataset, not the comparisons. */
        plot_ = new XYPlot( obsDataset_, xAxis, yAxis, null ) {
            public Range getDataRange( ValueAxis axis ) {
                if ( axis == xAxis ) {
                    return DatasetUtilities
                          .findDomainBounds( obsDataset_, true );
                }
                else if ( axis == yAxis ) {
                    return DatasetUtilities
                          .findRangeBounds( obsDataset_, true );
                }
                else {
                    return super.getDataRange( axis );
                }
            }
        };

        /* Configure and place the plot. */
        XYItemRenderer obsRenderer =
           new ErrorBarRenderer( false, true, true, true );
        obsRenderer.setPaint( Color.BLACK );
        plot_.setRenderer( obsRenderer );
        plot_.setDataset( 1, compDataset_ );
        plot_.setRenderer( 1, compDataset_.getRenderer() );
        JFreeChart chart = 
            new JFreeChart( null, JFreeChart.DEFAULT_TITLE_FONT, plot_,
                            true );
        add( new ChartPanel( chart ), BorderLayout.CENTER );
    }

    /**
     * Configures the observation and theory comparisons which are to be
     * displayed in this plot.
     *
     * @param  iobs  index of the observation within this dataset's
     *         ObservationSet
     * @param  comparisons   array of Comparison objects to be shown
     */
    public void setData( int iobs, Comparison[] comparisons ) {
        boolean[] selected = new boolean[ obsSet_.getObsCount() ];
        selected[ iobs ] = true;
        obsDataset_.setSelected( selected );
        compDataset_.setComparisons( comparisons );
        String xLabel = "";
        String yLabel = "";
        for ( int i = 0; i < comparisons.length && ( ! mismatchWarnX_ ||
                                                     ! mismatchWarnY_ ); i++ ) {
            TheorySet tset = comparisons[ i ].getTheorySet();
            MetadataItem ox = obsSet_.getMetadataX();
            MetadataItem oy = obsSet_.getMetadataY();
            MetadataItem tx = tset.getMetadataX();
            MetadataItem ty = tset.getMetadataY();
            if ( ! ox.isCompatible( tx ) && ! mismatchWarnX_ ) {
                logger_.warning( "X coords may not be compatible: "
                               + ox + " != " + tx );
                mismatchWarnX_ = true;
            }
            if ( ! oy.isCompatible( ty ) && ! mismatchWarnY_ ) {
                logger_.warning( "Y coords may not be compatible: "
                               + oy + " != " + ty );
                mismatchWarnY_ = true;
            }
        }
        revalidate();
    }

    /**
     * Returns the dataset used for observation plotting.
     *
     * @return  observation dataset
     */
    public ObservationSetDataset getObservationDataset() {
        return obsDataset_;
    }

    /**
     * Returns the dataset used for comparison plotting.
     *
     * @return  comparison dataset
     */
    public ComparisonDataset getComparisonDataset() {
        return compDataset_;
    }
}
