package fit.task;

import fit.util.TableObservationSetFactory;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Logger;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.MultiParameter;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.util.DataSource;

/**
 * Parameter for specifying the 'key' file which describes how a table
 * can be interpreted to turn it into an ObservationSet.
 *
 * <p>The format of this file (subject to change) currently looks like this:
 * <pre>
 * #  yColName     yerrColName        x    xWidth
 *    U            U_ERR            3600      800
 *    B            B_ERR            4500     1400
 *      ...
 * </pre>
 * (any comment lines starting with # are ignored).
 *
 * @author   Mark Taylor
 * @since    22 Nov 2006
 */
public class ObservationKeyParameter extends Parameter
                                     implements MultiParameter {

    private ObservationKey[] keys_;
    private static Logger logger_ = Logger.getLogger( "fit.task" );
    private static final char VAL_SEP = ',';

    /**
     * Constructor.
     *
     * @param  name  parameter name
     */
    public ObservationKeyParameter( String name ) {
        super( name );
        setUsage( "<key-file>" );
        setPrompt( "Location of observation key file" );
    }

    public char getValueSeparator() {
        return VAL_SEP;
    }

    public void setValueFromString( Environment env, String sval )
            throws TaskException {
        String[] svals = sval.split( "\\Q" + VAL_SEP + "\\E" );
        int nkey = svals.length;
        List keyList = new ArrayList();
        for ( int i = 0; i < nkey; i++ ) {
            InputStream in = null;
            try {
                in = DataSource.getInputStream( svals[ i ] );
                keyList.addAll( Arrays.asList( readKeys( in, svals[ i ] ) ) );
            }
            catch ( IOException e ) {
                throw new TaskException( "Error reading key file " + sval, e );
            }
            finally {
                if ( in != null ) {
                    try {
                        in.close();
                    }
                    catch ( IOException e ) {
                        // no action
                    }
                }
            }
        }
        keys_ = (ObservationKey[]) keyList.toArray( new ObservationKey[ 0 ] );
        super.setValueFromString( env, sval );
    }

    /**
     * Configures a TableObservationSetFactory according to the key file
     * represented by the value of this parameter.
     * This puts the factory in a position to generate an ObservationSet.
     * It is not an error for some of the keys in this parameter's value's
     * set to be absent from the factory's table, but if none of them are
     * present a TaskException will be thrown (since no sensible observation
     * set can then be generated).
     *
     * @param   env  execution environment
     * @param   fact   factory to configure
     */
    public void configureFactory( Environment env,
                                  TableObservationSetFactory fact )
            throws TaskException {
        checkGotValue( env );
        int nUse = 0;
        for ( int ikey = 0; ikey < keys_.length; ikey++ ) {
            ObservationKey key = keys_[ ikey ];
            double x = key.x_;
            double xWidth = key.xWidth_;
            String yColName = key.yColName_;
            String yerrColName = key.yerrColName_;
            String colPair = yColName + " +- " + yerrColName;
            try {
                fact.definePoint( x, xWidth * 0.5, yColName, yerrColName );
                logger_.info( "Using columns: " + colPair );
                nUse++;
            }
            catch ( IllegalArgumentException e ) {
                logger_.info( "Not using columns: " + colPair
                            + " (" + e + ")" );
            }
        }
        if ( nUse == 0 ) {
            throw new TaskException( "No samples defined by key file "
                                   + stringValue( env ) );
        }
    }

    /**
     * Parses the key file to generate an array of keys.
     *
     * @param  in  input stream
     * @param  file  filename 
     */
    private ObservationKey[] readKeys( InputStream in, String file )
            throws TaskException, IOException {
        BufferedReader rdr = new BufferedReader( new InputStreamReader( in ) );
        List keyList = new ArrayList();
        for ( String line; ( line = rdr.readLine() ) != null; ) {
            if ( line.startsWith( "#" ) ) {
                continue;
            }
            String[] words = line.trim().split( "\\s+" );
            if ( words.length != 4 ) {
                throw new TaskException( "Bad key file format " + file + " ("
                                       + words.length + " words on line)" );
            }
            String yColName = words[ 0 ];
            String yerrColName = words[ 1 ];
            double x;
            try {
                x = Double.parseDouble( words[ 2 ] );
            }
            catch ( NumberFormatException e ) {
                throw new TaskException( "Bad numeric value " + words[ 2 ]
                                       + " in key file " + file );
            }
            double xWidth;
            try { 
                xWidth = Double.parseDouble( words[ 3 ] );
            }
            catch ( NumberFormatException e ) {
                throw new TaskException( "Bad numeric value " + words[ 3 ]
                                       + " in key file " + file );
            }
            keyList.add( new ObservationKey( yColName, yerrColName,
                                             x, xWidth ) );
        }
        return (ObservationKey[]) keyList.toArray( new ObservationKey[ 0 ] );
    }

    /**
     * Helper class which describes each key of a key description file.
     * An instance of this class corresponds to one point of an ObservationSet.
     */
    private static class ObservationKey {
        final String yColName_;
        final String yerrColName_;
        final double x_;
        final double xWidth_;

        /**
         * Constructor.
         *
         * @param   yColName  column name for Y value
         * @param   yerrColName  column name for error on Y value
         * @param   x   central/nominal X value at which Y value is sampled
         * @param   xWidth  nominal range of X over which sample is taken
         */
        ObservationKey( String yColName, String yerrColName,
                        double x, double xWidth ) {
            yColName_ = yColName;
            yerrColName_ = yerrColName;
            x_ = x;
            xWidth_ = xWidth;
        }

        public String toString() {
            return x_ + " +- " + xWidth_ / 2.0 + ";\t"
                 + yColName_ + " +- " + yerrColName_;
        }
    }
}
