package fit.task;

import fit.framework.TheorySet;
import fit.gui.TheorySetDisplay;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.Executable;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.SingleTaskInvoker;
import uk.ac.starlink.task.Task;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.task.TerminalEnvironment;

/**
 * Task which does graphical display of one or more input TheorySets.
 *
 * @author   Mark Taylor
 * @since    22 Jan 2007
 */
public class PlotTheoryTask implements Task {

    private final TheorySetParameter tsetParam_;

    /**
     * Constructor.
     */
    public PlotTheoryTask() {
        tsetParam_ = new TheorySetParameter( "in" );
        tsetParam_.setPrompt( "Location of input model data" );
        tsetParam_.getReaderParameter().setName( "ifmt" );
        tsetParam_.setDescription( new String[] {
            "<p>File containing the model data to plot.",
            "The format is given by the",
            "<code>" + tsetParam_.getReaderParameter().getName() + "</code>",
            "parameter.",
            "Multiple model files (in the same format) may be specified",
            "by giving this parameter multiple times.",
            "</p>",
        } );
    }

    public String getPurpose() {
        return "Plots model data";
    }

    public Parameter[] getParameters() {
        return new Parameter[] {
            tsetParam_,
            tsetParam_.getReaderParameter(),
        };
    }

    public Executable createExecutable( Environment env ) throws TaskException {
        final TheorySet tset = tsetParam_.theorySetValue( env );
        return new Executable() {
            public void execute() {
                SwingUtilities.invokeLater( new Runnable() {
                    public void run() {
                        display( tset );
                    }
                } );
            }
        };
    }

    /**
     * Does display of a given TheorySet.  Must be called from the 
     * event dispatch thread.
     *
     * @param   tset   theory set to display
     */
    private static void display( TheorySet tset ) {
        JFrame frame = new JFrame( "Model data" );
        JComponent display = new TheorySetDisplay( tset );
        frame.getContentPane().add( display );
        frame.setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
        frame.pack();
        frame.setVisible( true );
    }

    /**
     * Invokes the task.  Try "-help".
     */
    public static void main( String[] args ) {
        TerminalEnvironment.NUM_TRIES = 1;
        new SingleTaskInvoker( new PlotTheoryTask(), "plotmodels" )
           .invoke( args );
    }
}
