package fit.gui;

import fit.framework.ObservationSet;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Paint;
import java.awt.Shape;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableModel;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.plot.DatasetRenderingOrder;
import org.jfree.chart.plot.XYPlot;

/**
 * Component which displays an ObservationSet in both a graphical panel
 * and a table.
 *
 * @author   Mark Taylor
 * @since    2 Feb 2007
 */
public class ObservationSetDisplay extends JPanel {

    private final JTable table_;
    private final JPanel plotPanel_;
    private ChartPanel plot_;
    private ObservationSetDataset selectionDataset_;

    /**
     * Constructor.
     */
    public ObservationSetDisplay() {
        super( new BorderLayout() );

        /* Prepare a panel ready to contain the graphical display. */
        plotPanel_ = new JPanel( new BorderLayout() );

        /* Prepare a table ready to contain the tabular display. */
        table_ = new JTable();
        JScrollPane scroller = new JScrollPane( table_ );
        JPanel tablePanel = new JPanel( new BorderLayout() );
        tablePanel.add( scroller, BorderLayout.CENTER );

        /* Arrange them in a split pane. */
        JSplitPane splitter = new JSplitPane( JSplitPane.VERTICAL_SPLIT );
        splitter.setTopComponent( scroller );
        splitter.setBottomComponent( plotPanel_ );
        splitter.setDividerLocation( 200 );
        add( splitter );
        setPreferredSize( new Dimension( 900, 900 ) );

        /* Prepare actions when rows are selected in the table display. */
        final ListSelectionModel selModel = table_.getSelectionModel();
        selModel.setSelectionMode( ListSelectionModel
                                  .MULTIPLE_INTERVAL_SELECTION );
        selModel.addListSelectionListener( new ListSelectionListener() {
            public void valueChanged( ListSelectionEvent evt ) {
                if ( selModel.getValueIsAdjusting() ) {
                    return;
                }
                int nobs = selectionDataset_.getObservationSet().getObsCount();
                boolean[] selected = new boolean[ nobs ];
                for ( int iobs = 0; iobs < nobs; iobs++ ) {
                    selected[ iobs ] = selModel.isSelectedIndex( iobs );
                }
                selectionDataset_.setSelected( selected );
                plotPanel_.repaint();
            }
        } );
    }

    /**
     * Constructs a display and initialises it with a given ObservationSet.
     *
     * @param   obsSet  initial observation set
     */
    public ObservationSetDisplay( ObservationSet obsSet ) {
        this();
        setObservationSet( obsSet );
    }

    /**
     * Sets the ObservationSet to be displayed in this component.
     *
     * @param   obsSet  observation set
     */
    public void setObservationSet( ObservationSet obsSet ) {

        /* Install tabular representation. */
        TableModel tableModel = new ObservationSetTableModel( obsSet );
        table_.setModel( tableModel );
        table_.setAutoResizeMode( tableModel.getColumnCount() == 1
                                ? JTable.AUTO_RESIZE_ALL_COLUMNS
                                : JTable.AUTO_RESIZE_OFF );

        /* Remove any existing plot from the display. */
        if ( plot_ != null ) {
            plotPanel_.remove( plot_ );
        }

        /* Set up axes. */
        final NumericAxis xAxis =
            new NumericAxis( obsSet.getMetadataX().toString(), false,
                             obsSet.getX( 0 ) );
        final NumericAxis yAxis =
            new NumericAxis( obsSet.getMetadataY().toString(), false,
                             obsSet.getY( 0, 0 ) );

        /* Set up primary dataset.  This contains all the observations in
         * a range of colours. */
        ObservationSetDataset dataset = new ObservationSetDataset( obsSet );
        final int nobs = obsSet.getObsCount();
        final ErrorBarRenderer renderer =
                new ErrorBarRenderer( true, true, true, true ) {
            public Paint getSeriesPaint( int is ) {
                float f = (float) is / (float) nobs;
                float alpha = 0.5f;
                return new Color( f, 0.5f, 1f - f, alpha );
            }
        };
        XYPlot xyplot = new XYPlot( dataset, xAxis, yAxis, null );
        xyplot.setRenderer( renderer );

        /* Set up secondary dataset.  This contains only those observations
         * which have been highlighted by the user, in black.  Initially
         * none are selected. */
        selectionDataset_ = new ObservationSetDataset( obsSet );
        selectionDataset_.setSelected( new boolean[ 0 ] );
        xyplot.setDataset( 1, selectionDataset_ );
        ErrorBarRenderer selectionRenderer =
                new ErrorBarRenderer( true, true, true, true ) {
            public Paint getSeriesPaint( int is ) {
                return Color.BLACK;
            }
            public Shape getItemShape( int is, int ic ) {
                return renderer
                      .getItemShape( selectionDataset_.getObsIndex( is ), ic );
            }
        };
        xyplot.setRenderer( 1, selectionRenderer );

        /* Arrange that the selection dataset is plotted over the primary one.
         * This means that it will not be obscured. */
        xyplot.setDatasetRenderingOrder( DatasetRenderingOrder.FORWARD );

        /* Install the new plot. */
        plot_ = new ChartPanel( new JFreeChart( null,
                                                JFreeChart.DEFAULT_TITLE_FONT,
                                                xyplot, false ) );
        plotPanel_.add( plot_, BorderLayout.CENTER );
    }
}
