package fit.test;

import fit.framework.FitCalculator;
import fit.util.ChiSquared;
import fit.util.LeastSquares;
import fit.util.Poisson;
import java.util.Random;
import junit.framework.TestCase;

public class FitCalculatorTest extends TestCase {

    private final Random random_;

    public FitCalculatorTest( String name ) {
        super( name );
        random_ = new Random( 42397L );
    }

    public void testCalcs() {
        exerciseCalculator( new LeastSquares() );
        exerciseCalculator( new ChiSquared( 1 ) );
        exerciseCalculator( new Poisson( 1 ) );
    }

    private void exerciseCalculator( FitCalculator fitCalc ) {
        double[] scales = { 1.0, Math.PI, Math.E, 99e99, 1e-23, -1.0, };
        int nscale = scales.length;
 
        int np = 50;
        double[] about1s = new double[ nscale ];
        double[] scores = new double[ nscale ];
        for ( int iscale = 0; iscale < nscale; iscale++ ) {
            Data data = new Data( np, 1000, 10, scales[ iscale ],
                                  new Random( -2300005L ) );
            double[] y1s = data.y1s_;
            double[] y2s = data.y2s_;
            double[] errs = data.errs_;

            double calcScale =
                fitCalc.getScale( np, y1s, y2s, errs );
            about1s[ iscale ] = calcScale * scales[ iscale ];

            assertEquals( 0.0, fitCalc.getScore( np, y1s, y1s, errs ),
                          Float.MIN_VALUE );
            assertEquals( 0.0, fitCalc.getScore( np, y2s, y2s, errs ), 
                          Float.MIN_VALUE );

            double[] sy2s = new double[ np ];
            for ( int ip = 0; ip < np; ip++ ) {
                sy2s[ ip ] = y2s[ ip ] * calcScale;
            }
            scores[ iscale ] = fitCalc.getScore( np, y1s, sy2s, errs );
            assertTrue( scores[ iscale ] > 0.0 );
 
            if ( iscale != 0 ) {
                assertTrue( scales[ iscale ] != 1.0 );
                assertTrue( fitCalc.getScore( np, y1s, y2s, errs )
                            / scores[ iscale ] > 1e3 );
            }
        }

        assertEquals( 1.0, about1s[ 0 ], 1e-2 );
        for ( int iscale = 1; iscale < nscale; iscale++ ) {
            assertEquals( about1s[ 0 ], about1s[ iscale ], 1e-12 );
            assertEquals( scores[ 0 ], scores[ iscale ], 1e-12 );
        }
    }

    private static class Data {
        final int np_;
        final double[] y1s_;
        final double[] y2s_;
        final double[] errs_;

        Data( int np, double yRange, double eRange, double scale,
              Random random ) {
           np_ = np;
           y1s_ = new double[ np ];
           y2s_ = new double[ np ];
           errs_ = new double[ np ];
           for ( int i = 0; i < np; i++ ) {
               y1s_[ i ] = ( 1.0 + random.nextDouble() ) * yRange;
               errs_[ i ] = random.nextDouble() * eRange;
               y2s_[ i ] =
                   scale * ( y1s_[ i ] + random.nextGaussian() * errs_[ i ] );
           }
        }
    }
}
