package fit.formats;

import fit.framework.MetadataItem;
import fit.framework.Theory;
import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.ArraysTheorySet;
import fit.util.StarMetadataTable;
import fit.util.TheoryFactory;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import uk.ac.starlink.fits.FitsTableBuilder;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.RowListStarTable;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.TableBuilder;
import uk.ac.starlink.table.TableFormatException;
import uk.ac.starlink.table.TableSink;
import uk.ac.starlink.table.Tables;
import uk.ac.starlink.util.DataSource;
import uk.ac.starlink.votable.VOTableBuilder;

/**
 * TableReader which can deserialise a {@link fit.framework.TheorySet}
 * written in table format.
 * Each row of the table represents a single {@link fit.framework.Theory}.
 * The first two columns of the format are <code>double[]</code> array
 * types containing (matched) X and Y vectors respectively, and any
 * subsequent columns represent per-theory metadata.
 * This is more or less the "native" format for the fitting program.
 *
 * @author   Mark Taylor
 * @since    19 Jan 2007
 * @see  fit.formats.ModelTableTheoryWriter
 */
public class ModelTableTheoryReader implements TheoryReader {

    private final TableBuilder fitsBuilder_ = new FitsTableBuilder();
    private final TableBuilder votableBuilder_ = new VOTableBuilder();

    public TheorySet readTheories( DataSource datsrc ) throws IOException {
        try {
            return readTheories( datsrc, fitsBuilder_ );
        }
        catch ( TableFormatException e ) {
            try {
                return readTheories( datsrc, votableBuilder_ );
            }
            catch ( TableFormatException e2 ) {
                e2.initCause( e );
                throw new FileFormatException( e2.getMessage(), e2 );
            }
        }
    }

    private TheorySet readTheories( DataSource datsrc, TableBuilder handler )
            throws IOException {
        InputStream in = new BufferedInputStream( datsrc.getInputStream() );
        try {
            ReaderSink sink = new ReaderSink();
            handler.streamStarTable( in, sink, null );
            return sink.getTheorySet();
        }
        finally {
            in.close();
        }
    }

    /**
     * TableSink which can receive an appropriate table and build a 
     * {@link fit.framework.TheorySet} from it.
     */
    private static class ReaderSink implements TableSink {

        RowListStarTable metaTable_;
        List theoryList_;
        TheorySet theorySet_;
        MetadataItem xmeta_;
        MetadataItem ymeta_;
        final int NOT_META = 2;
        int irow_;

        public void acceptMetadata( StarTable table )
                throws TableFormatException {
            List infoList =
                new ArrayList( Arrays
                              .asList( Tables.getColumnInfos( table ) ) );
            if ( infoList.size() < 2 ) {
                throw new TableFormatException( "Not in model table format "
                                              + "(<2 columns)" );
            }
            ColumnInfo xInfo = (ColumnInfo) infoList.remove( 0 );
            ColumnInfo yInfo = (ColumnInfo) infoList.remove( 0 );
            xmeta_ = new MetadataItem( xInfo.getName(), xInfo.getUnitString() );
            ymeta_ = new MetadataItem( yInfo.getName(), yInfo.getUnitString() );
            if ( ! double[].class.equals( xInfo.getContentClass() ) ||
                 ! double[].class.equals( yInfo.getContentClass() ) ) {
                throw new TableFormatException( "Not in model table format "
                                              + "(first 2 columns must be "
                                              + "double[] arrays)" );
            }
            metaTable_ =
                new RowListStarTable( (ColumnInfo[])
                                      infoList.toArray( new ColumnInfo[ 0 ] ) );
            theoryList_ = new ArrayList();
        }

        public void acceptRow( Object[] row ) {
            String name = "t" + ( ++irow_ );
            TheoryFactory tfact = new TheoryFactory();
            double[] lambdas = (double[]) row[ 0 ];
            double[] fluxes = (double[]) row[ 1 ];
            int npoint = lambdas.length;
            for ( int ip = 0; ip < npoint; ip++ ) {
                tfact.addPoint( lambdas[ ip ], fluxes[ ip ] );
            }
            theoryList_.add( tfact.createTheory( name ) );
            int nmeta = metaTable_.getColumnCount();
            Object[] metaCells = new Object[ nmeta ];
            System.arraycopy( row, NOT_META, metaCells, 0, nmeta );
            metaTable_.addRow( metaCells );
        }

        public void endRows() throws IOException {
            theorySet_ =
                new ArraysTheorySet( (Theory[]) theoryList_
                                               .toArray( new Theory[ 0 ] ),
                                     xmeta_, ymeta_,
                                     new StarMetadataTable( metaTable_ ) );
            theoryList_ = null;
	}

        /**
         * Returns the TheorySet built by this sink.
         * Only call it after the sink has been used (after endRows).
         *
         * @return   theory set
         */
        public TheorySet getTheorySet() {
            return theorySet_;
        }
    }
}
