package fit.framework;

/**
 * Describes the result of comparing an observation with a given Theory.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class Comparison implements Comparable {

    private final Theory theory_;
    private final double scale_;
    private final double score_;
    private final TheorySet theorySet_;

    /**
     * Constructor. 
     *
     * @param   theory   theory against which the comparison was done
     * @param   scale   multiplicative scaling factor applied to the
     *                  theory prior to the comparison
     * @param   score  score for how good the fit is (zero is perfect)
     * @param   theorySet  theory set from which <code>theory</code> comes - 
     *          used for metadata, but may be null if there is no known set
     */
    public Comparison( Theory theory, double scale, double score,
                       TheorySet theorySet ) {
        theory_ = theory;
        scale_ = scale;
        score_ = score;
        theorySet_ = theorySet;
    }

    /**
     * Returns the theory object against which this comparison was done.
     * Note that this was modified by the scale factor prior to the comparison.
     *
     * @return  theory against which comparison was done
     */
    public Theory getTheory() {
        return theory_;
    }

    /**
     * Returns the scale factor which was applied to the theory prior to
     * the comparison.
     *
     * @return   scale factor
     */
    public double getScale() {
        return scale_;
    }

    /**
     * Returns the fitting score for this comparison.
     * Zero is a perfect fit.
     *
     * @return   non-negative fitting score
     */
    public double getScore() {
        return score_;
    }

    /**
     * Returns the TheorySet which provided this comparison's theory.
     * May be <code>null</code>.
     *
     * @return  theory set
     */
    public TheorySet getTheorySet() {
        return theorySet_;
    }

    /**
     * Compares objects according to their fitting score.
     * Note this comparison is not consistent with <code>equals</code>.
     */
    public int compareTo( Object o ) {
        return Double.compare( getScore(), ((Comparison) o).getScore() );
    }

    public String toString() {
        StringBuffer sbuf = new StringBuffer();
        sbuf.append( String.valueOf( (float) score_ ) );
        while ( sbuf.length() < 16 ) {
            sbuf.append( ' ' );
        }
        sbuf.append( " - " );
        sbuf.append( theory_.getName() );
        if ( scale_ != 1.0 ) {
            sbuf.append( " * " )
                .append( Float.toString( (float) scale_ ) );
        }
        return sbuf.toString();
    }
}
