package fit.test;

import fit.framework.Interpolator;
import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.ObservationSet;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.util.ArraysMetadataTable;
import fit.util.ArraysObservationSet;
import fit.util.ArraysTheory;
import fit.util.ArraysTheorySet;
import fit.util.InterpolatorFactory;
import fit.util.LinearInterpolator;
import java.io.IOException;
import java.util.Random;

/**
 * Uses a notional function to produce both a test theory set and a 
 * simulated observation set.
 *
 * @author   Mark Taylor
 * @since    16 Nov 2006
 */
public class TestData {

    private final Random random_;
    private InterpolatorFactory interpFact_ = LinearInterpolator.getFactory();
    private double stretchX_ = 100.0;
    private double obsNoise_ = 0.01;

    /**
     * Constructs a new TestData with an arbitrary random number generator.
     */
    public TestData() {
        this( new Random() );
    }

    /**
     * Constructs a new TestData with a given random number generator.
     *
     * @param   random  random number generator
     */
    public TestData( Random random ) {
        random_ = random;
    }

    /**
     * Constructs a new TestData with a random number seed.
     *
     * @param   seed  random seed
     */
    public TestData( long seed ) {
        this( new Random( seed ) );
    }

    /**
     * Sets the noise factor for observations.  Should be between 0 and 1.
     *
     * @param  noise  noise
     */
    public void setNoise( double noise ) {
        obsNoise_ = noise;
    }

    /**
     * Returns the noise factor for observations.
     *
     * @return  noise
     */
    public double getNoise() {
        return obsNoise_;
    }

    /**
     * Calculates a function of two parameters.
     * It is used here to describe a family of theory functions 
     * <code>f(x)</code> distinguished by a distortion parameter
     * <code>e</code>.
     * <code>e</code> should be in the range -1&lt;=e&lt;=+1, and
     * <code>x</code> can be expected to be in the range 0-1.
     *
     * @param   x  independent variable
     *             (positive, with maximum value of order 1)
     * @param   e  distortion parameter (in range -1..+1)
     * @return  function value
     */
    public double func( double x, double e ) {
        if ( x < 0 ) {
            throw new IllegalArgumentException();
        }
        if ( e < -1 || e > +1 ) {
            throw new IllegalArgumentException();
        }
        double eperiod = 0.5 * ( 1 + e / 2 );
        double speriod = 0.05 * ( 1 - e / 3 );
        double samplitude = 0.2;
        double f = 1
                 + Math.exp( x / eperiod )
                 + Math.sin( x / speriod ) * samplitude;
        if ( e < 0 ) {
            f += -e * x / f;
        }
        else {
            f *= Math.exp( - ( x - 0.5 ) * ( x - 0.5 ) * e );
        }
        return f;
    }

    /**
     * Returns an observation set corresponding to this theoretical model.
     * The different observations in the set will be calculated using
     * a regular grid of distortion parameters.
     *
     * @param  nobs  number of observations
     * @param  npoint  number of points per observation
     * @return  synthetic observation set
     */
    public ObservationSet getObservationSet( int nobs, int npoint ) {

        /* Calculate sample points (X values). */
        double[] x01s = new double[ npoint ];
        double[] xs = new double[ npoint ];
        double[] xerrs = new double[ npoint ];
        Interpolator[] interps = new Interpolator[ npoint ];
        for ( int ip = 0; ip < npoint; ip++ ) {
            double x01 = random_.nextDouble();
            x01s[ ip ] = x01;
            double x = stretchX_ * x01;
            double x01width = x01 / npoint * random_.nextDouble() / 4;
            double xwidth = x01width * stretchX_;
            xs[ ip ] = x;
            xerrs[ ip ] = xwidth;
            interps[ ip ] = interpFact_.getInterpolator( x, xwidth );
        }

        /* Calculate observation values at each sample point (Y values). */
        double[] distortions = new double[ nobs ];
        double[] xfactors = new double[ nobs ];
        double[] yfactors = new double[ nobs ];
        double[][] ys = new double[ npoint ][ nobs ];
        double[][] yerrs = new double[ npoint ][ nobs ];
        for ( int iobs = 0; iobs < nobs; iobs++ ) {
            double e = ( 2.0 * iobs / nobs ) - 1.0;
            double yfact = 100 * random_.nextDouble();
            double xfact =
                1 + Math.pow( Math.abs( random_.nextGaussian() ), 2 );
            distortions[ iobs ] = e;
            yfactors[ iobs ] = yfact;
            xfactors[ iobs ] = xfact;
            for ( int ip = 0; ip < npoint; ip++ ) {
                double x01 = x01s[ ip ];
                double y = func( x01 / xfact, e ) * yfact;
                double yerr = y * random_.nextDouble() * obsNoise_;
                ys[ ip ][ iobs ] = y + random_.nextGaussian() * yerr;
                yerrs[ ip ][ iobs ] = yerr;
            }
        }

        /* Record per-observation metadata. */
        MetadataTable meta = new ArraysMetadataTable( 
            new String[] { "DISTORT", "YFACT", "XFACT", },
            new Object[] { distortions, yfactors, xfactors, } );

        /* Return synthetic observation set. */
        return new ArraysObservationSet( npoint, nobs, xs, xerrs, interps,
                                         ys, yerrs, xfactors,
                                         MetadataItem.X, MetadataItem.Y, meta );
    }

    /**
     * Returns a TheorySet correponding to this theoretical model.
     * The different theories in the set will be calculated using
     * a regular grid of distortion parameters.
     *
     * @param  nth  number of theories to generate
     * @param   npoint  number of points per theory
     * @return  synthetic theory set
     */
    public TheorySet getTheorySet( int nth, int npoint ) {

        /* Calculate sample points (X value) and theory parameters. */
        Theory[] theories = new Theory[ nth ];
        double[] distortions = new double[ nth ];
        double[] x01s = new double[ npoint ];
        double[] xs = new double[ npoint ];
        for ( int ip = 0; ip < npoint; ip++ ) {
            double x01 = 8.0 * ip / (double) npoint;
            x01s[ ip ] = x01;
            xs[ ip ] = x01 * stretchX_;
        }
        
        /* Calculate theory predictions (Y values). */
        for ( int ith = 0; ith < nth; ith++ ) {
            int ith1 = ith + 1;
            double e = ( 2.0 * ith ) / nth - 1.0;
            double[] ys = new double[ npoint ];
            distortions[ ith ] = e;
            for ( int ip = 0; ip < npoint; ip++ ) {
                double x01 = x01s[ ip ];
                ys[ ip ] = func( x01, e );
            }
            theories[ ith ] = new ArraysTheory( "T-" + ith1, xs, ys );
        }

        /* Record model parameters. */
        MetadataTable meta = new ArraysMetadataTable(
            new String[] { "DISTORT", },
            new Object[] { distortions, } );

        /* Return synthetic theory set. */
        return new ArraysTheorySet( theories, MetadataItem.X, MetadataItem.Y,
                                    meta );
    }
}
