package fit.util;

import fit.framework.FitCalculator;

/**
 * FitCalculator implementation which uses a scoring metric based on
 * the square of distances between observed and modelled point (as
 * does {@link ChiSquared}, but without any concept of error.
 * The fitting score is scaled by the average value of the observed
 * Y values, to make it dimensionless.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class LeastSquares implements FitCalculator {

    public double getScore( int np, double[] y1s, double[] y2s,
                            double[] errs ) {
        double y1sum = 0.0;
        for ( int ip = 0; ip < np; ip++ ) {
            y1sum += y1s[ ip ];
        }
        double yScale = y1sum / np;
        double s2sum = 0.0;
        int nUse = 0;
        for ( int ip = 0; ip < np; ip++ ) {
            double y1 = y1s[ ip ];
            double y2 = y2s[ ip ];
            double s = ( y1 - y2 ) / yScale;
            if ( ! Double.isNaN( s ) && ! Double.isInfinite( s ) ) {
                nUse++;
                s2sum += s * s;
            }
        }
        return s2sum / nUse;
    }

    public double getScale( int np, double[] y1s, double[] y2s, 
                            double[] errs ) {
        double sum12 = 0.0;
        double sum22 = 0.0;
        for ( int ip = 0; ip < np; ip++ ) {
            double y1 = y1s[ ip ];
            double y2 = y2s[ ip ];
            if ( ! Double.isNaN( y1 ) && ! Double.isInfinite( y1 ) &&
                 ! Double.isNaN( y2 ) && ! Double.isInfinite( y2 ) ) {
                sum12 += y1 * y2;
                sum22 += y2 * y2;
            }
        }
        return sum12 / sum22;
    }
}
