package fit.util;

/**
 * FitCalculator implementation which uses chi squared fitting with 
 * error values obtained on the assumption of Poisson errors
 * (the error at each point is proportional to the square root of
 * the value at that point).  Any data error values supplied to the
 * fitting evaluation methods are ignored.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class Poisson extends ChiSquared {

    /**
     * Constructor.
     *
     * @param  nparam  number of free parameters fitted to the data 
     *                 prior to the <code>getScore</code> call
     * @see   ChiSquared#ChiSquared(int)
     */
    public Poisson( int nparam ) {
        super( nparam );
    }

    public double getScore( int np, double[] y1s, double[] y2s,
                            double[] errs ) {
        return super.getScore( np, y1s, y2s, getPoissonErrors( np, y1s ) );
    }

    public double getScale( int np, double[] y1s, double[] y2s,
                            double[] errs ) {
        return super.getScale( np, y1s, y2s, getPoissonErrors( np, y1s ) );
    }

    /** 
     * Returns the Poisson errors for a set of data Y values.
     * These are just the square roots of the (absoluted) values.
     *
     * @param   np   number of points
     * @param   ys   array of values
     * @return   array of errors corresponding to <code>ys</code>
     */
    private static double[] getPoissonErrors( int np, double[] ys ) {
        double[] errs = new double[ np ];
        for ( int i = 0; i < np; i++ ) {
            errs[ i ] = Math.sqrt( Math.abs( ys[ i ] ) );
        }
        return errs;
    }
}
