package fit.util;

import fit.framework.Interpolator;
import fit.framework.Theory;

/**
 * Partial Interpolator implemenation which interpolates by smoothing 
 * values over a given range.  Concrete subclasses have effectively to
 * provide a weighting function (smoothing kernel) to determine how 
 * the smoothing is done.  The interpolated value is a normalised 
 * integral over the product of the theory and the weighting function.
 * Note that the bounds ought not to be too wide - the <code>getY</code>
 * method of this class will return NaN if the weighting function 
 * extends beyond the defined value of the function being smoothed.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public abstract class Smoother implements Interpolator {

    /**
     * Returns the bounds over which the weighting function is non-zero.
     *
     * @param   xMean   central value for the smoothing
     * @return  2-element array giving (low,high) bounds of weighting
     *          function
     */
    public abstract double[] getBounds( double xMean );

    /**
     * Returns the value of the weighting function at a given point.
     * The effect of calling this with <code>xPoint</code> outside the
     * range defined by <code>getBounds(xMean)</code> is undefined.
     * Since the result will be normalised, an arbitrary multiplicative
     * factor may be applied to the return value.
     *
     * @param   xMean   central value for the weighting
     * @param   xPoint  X value at whicht the weighting is required
     * @return  value of the weighting function at <code>xPoint</code>
     */
    public abstract double getWeight( double xMean, double xPoint );

    public double getY( Theory theory, double x ) {

        /* GIGO. */
        if ( Double.isNaN( x ) ) {
            return Double.NaN;
        }

        /* Work out the theory data points which bracket the whole range
         * over which this smoothing is to take place.  If this takes us
         * out of the defined range of the theory, return a blank. */
        double[] bounds = getBounds( x );
        if ( ! ( bounds[ 0 ] <= bounds[ 1 ] ) ) {
            throw new IllegalArgumentException(
                "Bad bounds: " + bounds[ 0 ] + ", " + bounds[ 1 ] );
        }
        double xlo = bounds[ 0 ];
        double xhi = bounds[ 1 ];
        assert xhi >= x;
        assert xlo <= x;
        int ifloor = theory.getFloorIndex( xlo );
        int iceil = FitUtils.getCeilingIndex( theory, xhi );
        if ( ifloor < 0 || iceil >= theory.getCount() ) {
            return Double.NaN;
        }
        assert x >= theory.getX( ifloor );
        assert x <= theory.getX( iceil );

        /* We need to use different algorithms according to how many theory
         * points we are integrating over. */
        assert iceil >= ifloor;
        switch ( iceil - ifloor ) {

            /* X is exactly on a theory point, and range is zero. */
            case 0: {
                return theory.getY( ifloor );
            }

            /* The entire range falls between two adjacent theory points.
             * Linearly interpolate between the points. */
            case 1: {
                double x0 = theory.getX( ifloor );
                double x1 = theory.getX( iceil );
                double y0 = theory.getY( ifloor );
                double y1 = theory.getY( iceil );
                return y0 + ( y1 - y0 ) * ( x - x0 ) / ( x1 - x0 );
            }

            /* The entire range brackets only a single theory point.
             * Just use the value at that point. */
            case 2: {
                return theory.getY( ifloor + 1 );
            }

            /* The range brackets many theory points.
             * Sum over weight * y * dx for each one, and normalise by 
             * dividing by the area of the weighting curve. */
            default: {
                assert x != theory.getX( ifloor );
                assert x != theory.getX( iceil );
                int ilo = ifloor + 1;
                int ihi = iceil - 1;
                assert ihi > ilo;
                assert theory.getX( ilo ) >= getBounds( x )[ 0 ];
                assert theory.getX( ihi ) <= getBounds( x )[ 1 ];
                double integral = 0;
                double norm = 0;
                for ( int i = ilo; i < ihi; i++ ) {
                    double x0 = theory.getX( i );
                    double x1 = theory.getX( i + 1 );
                    double y0 = theory.getY( i );
                    double y1 = theory.getY( i + 1 );
                    double area = 0.5 * ( y0 + y1 ) * ( x1 - x0 );
                    double weight = getWeight( x, 0.5 * ( x0 + x1 ) );
                    norm += ( x1 - x0 ) * weight;
                    integral += area * weight;
                }
                return integral / norm;
            }
        }
    }
}
