package fit.util;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * Object which contains an X coordinate.
 * The standard methods <code>equals</code>, <code>hashCode</code>
 * and <code>compareTo</code> are implemented in such a way that
 * instances of this class order themselves in order of increasing
 * X value, and two instances are considered equal if they have the
 * same X value.  This behaviour is intended to be inherited, but 
 * make sure that this is the semantics you want for object equality
 * before you use it in this way.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
class XValue implements Comparable {

    private final double x_;

    /**
     * Constructor.
     *
     * @param   x   X value
     */
    XValue( double x ) {
        x_ = x;
    }

    public int compareTo( Object o ) {
        return Double.compare( x_, ((XValue) o).x_ );
    }

    public boolean equals( Object o ) {
        return o instanceof XValue && ((XValue) o).x_ == x_;
    }

    public int hashCode() {
        return Float.floatToIntBits( (float) x_ );
    }

    /**
     * Sorts a list of <code>XValue</code> objects and removes duplicates.
     *
     * @param   pointList  list of XValues - modified in place
     */
    public static void normaliseList( List xvalueList ) {
        Collections.sort( xvalueList );
        XValue last = null;
        for ( Iterator it = xvalueList.iterator(); it.hasNext(); ) {
            XValue current = (XValue) it.next();
            if ( current.equals( last ) ) {
                it.remove();
            }
            else {
                last = current;
            }
        }
    }
}
