c
      subroutine betmod(bet,thc)
c
c Subroutine betmod: Create look-up tables of beta model and 
c                    its convolution with the PRF
c
c Update history:
c    V1.0  16-Oct-1992  Original version
c    V2.0  31-Dec-1992  Add FT convolution calculation
c    V3.0  16-Sep-1993  Do using subroutines
c    V4.0  17-Sep-1993  Get sampling parameters carefully
c    V4.1  27-Oct-1993  Correct case of small angles (truncate)
c    V4.2  31-Oct-1993  Changed to use steam convolution if required
c    V4.3   1-Apr-1994  Remove some debug messages
c
      implicit none
c
c Arguments:
c    bet = r*4 = beta value
c    thc = r*4 = core radius (arcmin)
c
      real*4 bet,thc
c
c Control parameters
c
      include 'PARAMETERS.INC'
c
c Storage for result
c
      include 'MODEL.INC'
c
c Beam file to get sampling
c
      include 'BEAM.INC'
c
c Data file for energy weighting
c
      include 'DATA.INC'
c
c Subprogrammes:
c    conft  = subroutine   = convolution by Bessel transform
c    getbj0 = r*4 function = J0 Bessel function
c
      real*4   getbj0
      external conft,getbj0
c
c Local parameters:
c    IALG  = i*4 = convoution algorithm (0=RFT;1=integration)
c    TINYB = r*4 = cutoff level for beam
c    TINYM = r*4 = cutoff level for model
c
      integer*4  IALG
      real*4     TINYB,TINYM
      parameter (IALG =0      )
      parameter (TINYB=1.0e-12)
      parameter (TINYM=1.0e-08)
c
c Local variables:
c    amax    = r*4        = maximum angle/core radius
c    ang     = r*4        = angle/core radius
c    bsmin   = r*4        = sampling in real space implied by beam
c    dsmin   = r*4        = sampling in real space implied by data
c    epsf    = r*4        = PSF energy/keV
c    flux    = r*4        = summed flux/counts
c    i1      = i*4        = counting index
c    fang(i) = r*4(NMMAX) = angles for model and beam sampling
c    farg(i) = r*4(NMMAX) = wavenumbers for FT sampling
c    ftcd(i) = r*4(NMMAX) = fourier transform of convolution
c    lenft   = i*4        = length of FT arrays
c    pow     = r*4        = power for raising functions
c    sampl   = r*4        = real-space sampling of arrays
c    stepft  = r*4        = sampling in FT space implied by beam
c    xsmin   = r*4        = sampling in real space implied by model
c
      real*4    amax,ang,bsmin,dsmin,epsf,flux,pow,
     +          sampl,stepft,xsmin
      real*4    fang(NMMAX),farg(NMMAX),ftcd(NMMAX)
      integer*4 i1,lenft
c
      write (6,991)
      write (6,994) TINYB,TINYM
c
c Get number of points to use in beam and model tables and in
c    radial FT
c
      NMTAB=min0(NBMAX,NMMAX)
      NBTAB=min0(NBMAX,NMMAX)
      lenft=NMMAX
c
c Use declared beam type to get maximum sampling permitted
c    for beam in real space (arcmin) and the sampling in
c    FT space. Since the FT space sampling merely requires
c    that one of the beam FT and the model FT is sufficiently
c    small, this single FT sampling parameter is adopted without
c    reference to the model.
c
      if     (IDENTB .eq. 1) then
        bsmin =   0.002
        stepft= 316.2/float(lenft-1)
      elseif (IDENTB .eq. 2) then
        bsmin =   0.0002
        stepft=3162.0/float(lenft-1)
      else
        write (6,699)
        IDENTB=   1
        bsmin =   0.002
        stepft= 316.2/float(lenft-1)
      endif
c
c Use model parameters to get maximum sampling permitted
c    for model (arcmin)
c
      xsmin=0.01*thc
c
c Use data range to get maximum sampling permitted for
c    coverage of full region
c
      dsmin=1.5*DL2/float(NMTAB-1)
c
c Set sampling parameters of beam and model, reporting
c    choice.
c
      sampl=dsmin
      if (bsmin .lt. sampl) then
        write (6,697)
      endif
      if (xsmin .lt. sampl) then
        write (6,696)
      endif
      write (6,995) bsmin,dsmin,xsmin,sampl
c
c Copy real-space sampling into common blocks
c
      BSMP =sampl
      BSBM =sampl
c
c Copy maxima into common blocks
c
      BMAX =float(NBTAB-1)*BSMP
      BBMX =float(NMTAB-1)*BSBM
c
c Loop through angles, calculating basic beta model at each
c    angle, truncating calculation when the model values get
c    too small
c
      if (bet .gt. (1.0/6.0)) then
        pow =1.0/(1.0-6.0*bet)
        amax=TINYM**pow
      else
        amax=1.0/TINYM
        write (6,698) bet
      endif
c
      write (6,998) BSBM,BBMX,NMTAB
c
      pow=0.5-3.0*bet
      do i1=1,NMTAB
        ang=float(i1-1)*BSBM/thc
        if (ang .gt. amax) then
          BMDL(i1)=0.0
        else
          BMDL(i1)=(1.0+ang**2)**pow
        endif
      enddo
c
c Loop through angles, calculating beam value at each angle, 
c    truncating at values that are too small
c
      write (6,997) BSMP,BMAX,NBTAB
c
      if (NWTAB .eq. 0) then
c
        epsf = 1.0
c
        do i1=1,NBTAB
          ang=BSMP*float(i1-1)
          if     (IDENTB .eq. 1) then
            call rospspc(ang,epsf,BVAL(i1))
          elseif (IDENTB .eq. 2) then
            call roshri (ang,epsf,BVAL(i1))
          endif
        enddo
c
        flux=0.0
        do i1=1,NDTAB
          flux=flux+A(i1)*(D2(i1)**2-D1(i1)**2)
        enddo
        flux=flux*4.0*atan(1.0)*3600.0
c
        NWTAB=1
        WTS(1)=flux
        ENS(1)=epsf
c
      else
c
        do i1=1,NBTAB
          ang=BSMP*float(i1-1)
          call rospwt(ang,BVAL(i1),IDENTB)
        enddo
c
      endif
c
c Truncate beam table for very small values
c
      do i1=1,NBTAB
        if (BVAL(i1) .lt. TINYB) then
          BVAL(i1)=0.0
        endif
      enddo
c
c Calculate radius array
c
      do i1=1,NMTAB
        fang(i1)=BSBM*float(i1-1)
      enddo
c
c Get wavenumber space array
c
      do i1=1,lenft
        farg(i1)=stepft*float(i1-1)
      enddo
c
c Perform convolution
c
      if ((stepft*BBMX) .gt. 0.5) then
        write (6,695) (stepft*BBMX)
      endif
      write (6,999) lenft,stepft,(stepft*BBMX)
c
      if (IALG .le. 0) then
        call conft(fang,BMDL,BVAL,CMDL,NMTAB,farg,ftcd,lenft)
      else
        write (6,993)
        call convl(fang,BMDL,BVAL,CMDL,NMTAB)
      endif
c
c Calculation complete
c
      write (6,992)
      return
c
c Error messages
c
 695  format (' '/
     +        ' * Warning: betmod: FT safety parameter = ',f8.3,' *'/)
 696  format (' '/
     +        ' * Warning: betmod: xsmin less than sampl          *'/)
 697  format (' '/
     +        ' * Warning: betmod: bsmin less than sampl          *'/)
 698  format (' '/
     +        ' * Warning: betmod: beta = ',f8.3,' is unphysical  *'/)
 699  format (' '/
     +        ' * Warning: betmod: beam unset, ROSAT PSPC assumed *'/)
c
c Debug formats
c
 991  format (' '/
     +        ' *   Debug: betmod: messages start                 *'/)
 992  format (' *   Debug: betmod: messages end                   *'/)
 993  format (' *   Debug: betmod: using old-style convolution    *'/)
 994  format (' *   Debug: betmod: cutoff levels                  *'/
     +        '               beam = ',1pe15.5/
     +        '              model = ',1pe15.5)
 995  format (' *   Debug: betmod: samplings                      *'/
     +        '              bsmin = ',1pe15.5/
     +        '              dsmin = ',1pe15.5/
     +        '              xsmin = ',1pe15.5/
     +        '              sampl = ',1pe15.5,' used')
 997  format (' *   Debug: betmod: beam calculation starting      *'/
     +        '               BSMP = ',1pe15.5/
     +        '               BMAX = ',1pe15.5/
     +        '              NBTAB = ',0pi15  )
 998  format (' *   Debug: betmod: model calculation starting     *'/
     +        '               BSBM = ',1pe15.5/
     +        '               BBMX = ',1pe15.5/
     +        '              NMTAB = ',0pi15  )
 999  format (' *   Debug: betmod: RFT convolution starting       *'/
     +        '              lenft = ',    i15/
     +        '             stepft = ',1pe15.5/
     +        '      safety factor = ',1pe15.5)
c
      end
