package fit.task;

import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.FitUtils;
import java.io.IOException;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.MultiParameter;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.ParameterValueException;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.util.DataSource;

/**
 * Parameter for specifying a TheorySet.
 *
 * @author   Mark Taylor
 * @since    22 Nov 2006
 */
public class TheorySetParameter extends Parameter implements MultiParameter {

    private final TheoryReaderParameter readerParam_;
    private TheorySet theorySet_;
    private static final char VAL_SEP = ',';

    /**
     * Constructor.
     *
     * @param   name  parameter name
     */
    public TheorySetParameter( String name ) {
        super( name );
        setUsage( "<model-file>" );
        setPrompt( "File containing model data" );

        readerParam_ = new TheoryReaderParameter( name + "fmt" );
        readerParam_.setDefault( "ymodel" );
        readerParam_.setPrompt( "Format for model data " + name );
        readerParam_.setDescription( new String[] {
            "<p>Defines the format of the file given by the",
            "<code>" + name + "</code> parameter.",
            "See <ref id='modelfmt'/> for a list of known formats.",
            "</p>",
        } );
    }

    /**
     * Returns the associated parameter which can read the theory file format.
     *
     * @return   reader parameter
     */
    public TheoryReaderParameter getReaderParameter() {
        return readerParam_;
    }

    public char getValueSeparator() {
        return VAL_SEP;
    }

    public void setValueFromString( Environment env, String sval )
            throws TaskException {
        TheoryReader reader = readerParam_.readerValue( env );
        String[] svals = sval.split( "\\Q" + VAL_SEP + "\\E" );
        int nset = svals.length;
        TheorySet[] theorySets = new TheorySet[ nset ];
        for ( int i = 0; i < nset; i++ ) {
            try {
                DataSource datsrc = DataSource.makeDataSource( svals[ i ] );
                theorySets[ i ] = reader.readTheories( datsrc );
            }
            catch ( IOException e ) {
                throw new ParameterValueException( this, e );
            }
        }
        theorySet_ = FitUtils.concatTheorySets( theorySets );
        super.setValueFromString( env, sval );
    }

    /**
     * Returns the theory set specified by this parameter.
     *
     * @param   env  execution environment
     * @return  theory set
     */
    public TheorySet theorySetValue( Environment env ) throws TaskException {
        checkGotValue( env );
        return theorySet_;
    }
}
