package fit.util;

import fit.framework.Interpolator;
import fit.framework.ObservationSet;
import fit.framework.MetadataItem;
import fit.framework.MetadataTable;

/**
 * Abstract ObservationSet implementation which is based on an existing 
 * ObservationSet but manipulates its Y values.  This is suitable for
 * unit changes etc.  Concrete subclasses must provide a suitable 
 * implementation of {@link #transformY}.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public abstract class TransformerObservationSet implements ObservationSet {

    private final ObservationSet base_;
    private final MetadataItem metaY_;

    /**
     * Constructor.
     *
     * @param   base  base observation set
     */
    public TransformerObservationSet( ObservationSet base,
                                      MetadataItem metaY ) {
        base_ = base;
        metaY_ = metaY;
    }

    /**
     * Transforms a (y,y_error) pair from the base set to provide the
     * (y,y_error) pair for this one.
     *
     * @param   y  base Y value
     * @param   yerr  base Y error
     * @return  2-element array giving the transformed (y,yerr)
     */
    public abstract double[] transformY( double y, double yerr );

    public int getObsCount() {
        return base_.getObsCount();
    }

    public int getPointCount() {
        return base_.getPointCount();
    }

    public double getX( int ip ) {
        return base_.getX( ip );
    }

    public double getXError( int ip ) {
        return base_.getXError( ip );
    }

    public Interpolator getInterpolator( int ip ) {
        return base_.getInterpolator( ip );
    }

    public double getY( int ip, int iobs ) {
        return transformY( base_.getY( ip, iobs ),
                           base_.getYError( ip, iobs ) )[ 0 ];
    }

    public double getYError( int ip, int iobs ) {
        return transformY( base_.getY( ip, iobs ),
                           base_.getYError( ip, iobs ) )[ 1 ];
    }

    public double getXFactor( int iobs ) {
        return base_.getXFactor( iobs );
    }

    public MetadataItem getMetadataX() {
        return base_.getMetadataX();
    }

    public MetadataItem getMetadataY() {
        return metaY_;
    }

    public MetadataTable getMetadataTable() {
        return base_.getMetadataTable();
    }
}
