package fit.formats;

import java.io.IOException;

/**
 * Utility methods for format handlers.
 *
 * @author   Mark Taylor
 * @since    26 Oct 2006
 */
class FormatUtils {

    /**
     * Private sole constructor to prevent instantiation.
     */
    private FormatUtils() {
    }

    /**
     * Returns the trimmed content of a string minus a given prefix.
     *
     * @param  line  input line
     * @param  prefix   prefix to ignore
     * @return  the rest of line
     */
    public static String readTrail( String line, String prefix ) {
        return line.substring( prefix.length() ).trim();
    }

    /**
     * Returns an array of words in a given string excluding a given prefix.
     *
     * @param   line   input line
     * @param   prefix  prefix to ignore
     * @return   space-separated words in the non-ignored part of
     *           <code>line</code>
     */
    public static String[] readHeadings( String line, String prefix ) {
        return line.substring( prefix.length() ).trim().split( " +" );
    }

    /**
     * Reads a list of whitespace-separated floating point numbers from a 
     * string.  Any failure is thrown as an IOException with an informative
     * error message.
     *
     * @param  line   line of text
     * @param  line1  1-based line of text file, used for error messages
     * @param  nval  number of values which must be present (ignored if &lt;=0)
     * @throws  IOException  if numbers can't be parsed or the wrong number
     *          are present
     */
    public static double[] readNumbers( String line, int iline1, int nval )
            throws IOException {
        String[] words = line.trim().split( "\\s+" );
        int nword = words.length;
        if ( nval >= 0 && nword != nval ) {
            throw new IOException( "Expected " + nval + " values, got "
                                 + nword + " at line " + iline1 );
        }
        double[] values = new double[ nword ];
        for ( int iword = 0; iword < nword; iword++ ) {
            String word = words[ iword ];
            double val;
            try {
                val = Double.parseDouble( words[ iword ] );
            }
            catch ( NumberFormatException e ) {
                throw new IOException( "Bad number \"" + word +
                                       "\" at line " + iline1 );
            }
            if ( Double.isNaN( val ) || Double.isInfinite( val ) ) {
                throw new IOException( "Non-finite number " + word +
                                       " at line " + iline1 );
            }
            values[ iword ] = val;
        }
        return values;
    }
}
