package fit.gui;

import fit.framework.Comparison;
import fit.framework.ObservationSet;
import fit.framework.TheorySet;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.ListSelectionModel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableModel;
import org.jfree.chart.LegendItem;

/**
 * Component which displays the results of a set of observation/theory
 * comparisons.
 *
 * @author   Mark Taylor
 * @since    31 Oct 2006
 */
public class Display extends JPanel {

    private final JTable obsTable_;
    private final JTable compTable_;
    private final JPanel plotPanel_;
    private final ColDef legendColDef_;
    private ComparisonTableModel dummyCompModel_;
    private ResultPlot plot_;
    private ComparisonTableModel[] compModels_;
    private TheorySet theorySet_;

    /**
     * Constructor.
     */
    public Display() {
        super( new BorderLayout() );

        /* Set up a table to hold observation data. */
        obsTable_ = new JTable();

        /* Set up a column to hold plot symbols in the comparison table. */
        legendColDef_ = new ColDef( null, Icon.class ) {
            public Object getValueAt( int irow ) {
                Comparison comp = ((ComparisonTableModel) compTable_.getModel())
                                 .getComparison( irow );
                LegendItem legend =
                    plot_.getComparisonDataset().getLegendItem( comp );
                return legend == null ? null : new LegendIcon( legend );
            }
        };

        /* Set up a table to hold comparison data. */
        compTable_ = new RenderTweakTable( new ComparisonRenderTweaker() );
        setTheorySet( null );

        /* Set up a plot to display fitted points and curves. */
        plotPanel_ = new JPanel( new BorderLayout() );
 
        /* Place the observation table in a scroller. */
        JScrollPane obsScroller = new JScrollPane( obsTable_ );
        JPanel obsPanel = new JPanel( new BorderLayout() );
        obsPanel.add( obsScroller, BorderLayout.CENTER );
        obsPanel.setBorder( titleBorder( "Observations" ) );

        /* Place the comparisons table in a scroller. */
        JScrollPane compScroller = new JScrollPane( compTable_ );
        final JPanel compPanel = new JPanel( new BorderLayout() );
        compPanel.add( compScroller, BorderLayout.CENTER );
        compPanel.setBorder( titleBorder( "Models" ) );

        /* Arrange the main components in split panes and add to this
         * container. */
        JSplitPane tSplitter = new JSplitPane();
        tSplitter.setLeftComponent( obsPanel );
        tSplitter.setRightComponent( compPanel );
        tSplitter.setDividerLocation( 240 );
        JSplitPane gSplitter = new JSplitPane( JSplitPane.VERTICAL_SPLIT );
        gSplitter.setLeftComponent( tSplitter );
        gSplitter.setRightComponent( plotPanel_ );
        gSplitter.setDividerLocation( 200 );
        add( gSplitter );
        setPreferredSize( new Dimension( 900, 900 ) );

        /* Set up the action for selecting an observation. */
        final ListSelectionModel obsSelModel = obsTable_.getSelectionModel();
        obsSelModel.setSelectionMode( ListSelectionModel.SINGLE_SELECTION );
        obsSelModel.addListSelectionListener( new ListSelectionListener() {
            public void valueChanged( ListSelectionEvent evt ) {
                int isel = obsSelModel.getMinSelectionIndex();
                assert obsSelModel.getMaxSelectionIndex() == isel;
                obsSelected( isel );
            }
        } );

        /* Set up the action for selecting comparisons. */
        final ListSelectionModel compSelModel = compTable_.getSelectionModel();
        compSelModel.setSelectionMode( ListSelectionModel
                                      .MULTIPLE_INTERVAL_SELECTION );
        compSelModel.addListSelectionListener( new ListSelectionListener() {
            public void valueChanged( ListSelectionEvent evt ) {
                if ( compSelModel.getValueIsAdjusting() ) {
                    return;
                }
                int iobs = obsSelModel.getMinSelectionIndex();
                assert obsSelModel.getMaxSelectionIndex() == iobs;
                List cList = new ArrayList();
                for ( int i = compSelModel.getMinSelectionIndex();
                      i <= compSelModel.getMaxSelectionIndex(); i++ ) {
                    if ( compSelModel.isSelectedIndex( i ) ) {
                        cList.add( compModels_[ iobs ].getComparison( i ) );
                    }
                }
                Comparison[] comps =
                    (Comparison[]) cList.toArray( new Comparison[ 0 ] );
                plot_.setData( iobs, comps );
                compPanel.repaint();
            }
        } );
    }

    /**
     * Sets the ObservationSet to be displayed by this component.
     *
     * @param  obsSet  observation set
     */
    public void setObservationSet( ObservationSet obsSet ) {
        if ( plot_ != null ) {
            plotPanel_.remove( plot_ );
        }
        plot_ = new ResultPlot( obsSet );
        TableModel obsModel = new ObservationSetTableModel( obsSet );
        obsTable_.setModel( obsModel );
        obsTable_.setAutoResizeMode( obsModel.getColumnCount() == 1 
                                   ? JTable.AUTO_RESIZE_ALL_COLUMNS
                                   : JTable.AUTO_RESIZE_OFF );
        int nobs = obsSet.getObsCount();
        compModels_ = new ComparisonTableModel[ nobs ];
        plotPanel_.add( plot_, BorderLayout.CENTER );
    }

    /**
     * Returns the ObservationSet currently displayed by this component.
     *
     * @return  observation set
     */
    public ObservationSet getObservationSet() {
        TableModel model = obsTable_.getModel();
        return model instanceof ObservationSetTableModel
             ? ((ObservationSetTableModel) model).getObservationSet()
             : null;
    }

    /**
     * Sets the TheorySet from which the comparisons to be displayed by this
     * component are taken.  May be null.
     *
     * @param  theorySet  theory set
     */
    public void setTheorySet( TheorySet theorySet ) {
        theorySet_ = theorySet;
        compTable_.setAutoCreateColumnsFromModel( true );
        dummyCompModel_ =
            createComparisonTableModel( legendColDef_, theorySet );
        compTable_.setModel( dummyCompModel_ );
        compTable_.setAutoCreateColumnsFromModel( false );
        compTable_.setAutoResizeMode( theorySet == null
                                    ? JTable.AUTO_RESIZE_ALL_COLUMNS
                                    : JTable.AUTO_RESIZE_OFF );
    }

    /**
     * Returns the TheorySet from which the comparisons displayed by this
     * component are taken, if known.
     *
     * @return   current theory set, or null
     */
    public TheorySet getTheorySet() {
        return theorySet_;
    }

    /**
     * Adds a new comparison object to be displayed by this component.
     *
     * @param   iobs   index of the observation to which this result pertains
     * @param   comp   comparison of observation <code>iobs</code> with 
     *          a certain theory
     */
    public void addResult( int iobs, Comparison comp ) {
        if ( compModels_[ iobs ] == null ) {
            compModels_[ iobs ] =
                createComparisonTableModel( legendColDef_, theorySet_ );
        }
        compModels_[ iobs ].addComparison( comp );
    }

    /**
     * Indicates that some or all of the results have been added to this
     * display.
     */
    public void dataLoaded() {

        /* Select the first result so that the display is initialised 
         * rather than waiting for the user to make a selection before
         * anything useful is displayed. */
        ListSelectionModel obsSelModel = obsTable_.getSelectionModel();
        if ( obsSelModel.getMinSelectionIndex() == -1 ) {
            obsSelModel.addSelectionInterval( 0, 0 );
        }
    }

    /**
     * Invoked when one of the observations in the current observation
     * set has been selected by the user.
     *
     * @param   isel   index of selected observation
     */
    private void obsSelected( int isel ) {

        /* No selection - display empty comparison. */
        if ( isel < 0 ) {
            compTable_.setModel( dummyCompModel_ );
            plot_.setData( -1, new Comparison[ 0 ] );
        }

        /* Selection of observation with no known comparisons - display
         * empty comparison. */
        else if ( compModels_[ isel ] == null ) {
            compTable_.setModel( dummyCompModel_ );
            plot_.setData( isel, new Comparison[ 0 ] );
        }

        /* Otherwise, there are comparisons to display. */
        else {

            /* Display the set of comparisons. */
            ComparisonTableModel compModel = compModels_[ isel ];
            compTable_.setModel( compModel );

            /* Locate the best fit row. */
            int ibest = -1;
            double scoreBest = Double.NaN;
            for ( int i = 0; i < compModel.getRowCount(); i++ ) {
                double score = compModel.getComparison( i ).getScore();
                if ( ! ( scoreBest < score ) ) {
                    scoreBest = score;
                    ibest = i;
                }
            }

            /* Highlight it. */
            if ( ! Double.isNaN( scoreBest ) ) {
                selectRow( compTable_, ibest );
            }
        }
    }

    /**
     * Creates a titled border.
     *
     * @param  title  border title
     * @return  border
     */
    private static Border titleBorder( String title ) {
        Border lineBorder = BorderFactory.createLineBorder( Color.BLACK );
        return BorderFactory.createTitledBorder( lineBorder, title );
    }

    /**
     * Selects a given row in a table and scrolls to make it visible.
     *
     * @param  table  table
     * @param  irow   row index
     */
    private static void selectRow( JTable table, int irow ) {

        /* If the table is in a scrollpane, scroll it to make the row 
         * visible.  Note that JViewport.scrollRectToVisible() doesn't
         * do what you think it should. */
        Component parent = table.getParent();
        if ( parent instanceof JViewport ) {
            JViewport vp = (JViewport) parent;
            Rectangle rect = table.getCellRect( irow, -1, true );
            int targetY = rect.y + rect.height / 2;
            int viewHeight = vp.getExtentSize().height;
            int topY = targetY - viewHeight / 2;
            int y =
                Math.max( 0, Math.min( topY, table.getHeight() - viewHeight ) );
            vp.setViewPosition( new Point( 0, y ) );
        }

        /* Set the single row selection to the chosen row. */
        table.setRowSelectionInterval( irow, irow );
    }

    /**
     * Constructs and returns a new ComparisonTableModel.
     * An additional column, which displays the plotting symbol currently
     * being used for the comparison (if any), is also added.
     *
     * @param   coldef0  column to insert at the start
     * @param   theorySet  theory set from which comparison theories are taken,
     *          or null
     * @return   modified comparison table model
     */
    private static ComparisonTableModel 
            createComparisonTableModel( ColDef coldef0, TheorySet theorySet ) {
        ComparisonTableModel compModel = new ComparisonTableModel( theorySet );
        List colDefList =
            new ArrayList( Arrays.asList( compModel.getColDefs() ) );
        if ( coldef0 != null ) {
            colDefList.add( 0, coldef0 );
        }
        compModel.setColDefs( (ColDef[])
                              colDefList.toArray( new ColDef[ 0 ] ) );
        return compModel;
    }

    /**
     * Icon which displays a LegendItem.
     */
    private static class LegendIcon implements Icon {
        private final LegendItem legend_;
        private final Map legendHints_;

        /**
         * Constructor.
         *
         * @param  legend   legend item to display
         */
        public LegendIcon( LegendItem legend ) {
            legend_ = legend;
            legendHints_ = new HashMap();
            legendHints_.put( RenderingHints.KEY_ANTIALIASING,
                              RenderingHints.VALUE_ANTIALIAS_ON );
        }

        public int getIconWidth() {
            return legend_.getShape().getBounds().width;
        }

        public int getIconHeight() {
            return legend_.getShape().getBounds().height;
        }

        public void paintIcon( Component c, Graphics g, int x, int y ) {
            Graphics2D g2 = (Graphics2D) g.create();
            g2.translate( x, y );
            g2.addRenderingHints( legendHints_ );
            if ( legend_.isShapeOutlineVisible() ) {
                g2.setPaint( legend_.getOutlinePaint() );
                g2.draw( legend_.getShape() );
            }
            if ( legend_.isShapeFilled() ) {
                g2.setPaint( legend_.getFillPaint() );
                g2.fill( legend_.getShape() );
            }
        }
    }

    /**
     * Modifies a rendered table cell which will display a Comparison object.
     * The idea is to make it visually obvious how good the fitting score is.
     */
    private static class ComparisonRenderTweaker implements RenderTweaker {

        public void tweakComponent( Component comp, JTable table, Object value,
                                    boolean isSelected, boolean hasFocus,
                                    int irow, int icol ) {
            TableModel model = table.getModel();
            if ( model instanceof ComparisonTableModel ) {
                ComparisonTableModel cmodel = (ComparisonTableModel) model;
                Comparison comparison = cmodel.getComparison( irow );
                double[] range = cmodel.getScoreRange();
                double wrongness = comparison.getScore() == 0.0
                                 ? 0.0
                                 : ( comparison.getScore() - range[ 0 ] )
                                     / ( range[ 1 ] - range[ 0 ] );
                wrongness = Math.pow( wrongness, 0.5 );
                assert wrongness >= 0 && wrongness <= 1;

                /* Adjust the alpha (transparency) of the foreground colour
                 * according to how good the fit is.  Possibly there are
                 * visually better ways of doing this.  A variably coloured
                 * background might be nice, but one has to worry about 
                 * how highlighting selection is going to interact with it. */
                int fgRgb = table.getForeground().getRGB();
                int maxFaint = 0xd0;
                fgRgb = ( fgRgb & 0x00ffffff )
                      | ( (int) ( maxFaint * ( 1 - wrongness ) +
                                             ( 0xff - maxFaint ) ) << 24 );
                comp.setForeground( new Color( fgRgb, true ) );

                /* For the Symbol column, the background doesn't get
                 * highlighted on selection. */
                if ( icol == 0 && isSelected ) {
                    comp.setBackground( table.getBackground() );
                }
            }
        }
    }
}
