package fit.util;

import fit.framework.Interpolator;
import fit.framework.Theory;

/**
 * Interpolator implementation which uses linear interpolation.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class LinearInterpolator implements Interpolator {

    public double getY( Theory theory, double x ) {
        int im = theory.getFloorIndex( x );

        /* If x is smaller than lowest X coordinate in the sequence,
         * don't attempt to extrapolate. */
        if ( im == -1 ) {
            return Double.NaN;
        }

        /* If x is equal to a value in the sequence, return it. */
        double xm = theory.getX( im );
        if ( xm == x ) {
            return theory.getY( im );
        }
    
        /* If x is greater than the highest X coordinate in the sequence,
         * don't attempt to extrapolate. */
        if ( im == theory.getCount() - 1 ) {
            return Double.NaN;
        }

        /* Otherwise, interpolate. */
        int ip = im + 1;
        double xp = theory.getX( ip );
        double ym = theory.getY( im );
        double yp = theory.getY( ip );
        return ym + ( x - xm ) / ( xp - xm ) * ( yp - ym );
    }

    
    /**
     * Returns an InterpolatorFactory which dispenses
     * <code>LinearInterpolator</code>s.
     */
    public static InterpolatorFactory getFactory() {
        return new InterpolatorFactory() {
            public Interpolator getInterpolator( double x, double xerr ) {
                return new LinearInterpolator();
            }
        };
    }
}
