package fit.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import fit.framework.Theory;

/**
 * Constructs a Theory object from individual point specifications.
 * Use this class by making multiple calls to {@link #addPoint} 
 * giving (x,y) pairs, and then calling <code>createTheory</code>.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class TheoryFactory {

    private final List itemList_ = new ArrayList();

    /**
     * Adds a new point to the theory that this factory will create.
     * There are no constraints on what order this method is called in
     * for different points (it is not necessary to call it in 
     * increasing order of <code>x</code>.  Duplicate or NaN values
     * will be ignored.
     *
     * @param   x  x coordinate
     * @param   y  y coordinate
     */
    public void addPoint( double x, double y ) {
        if ( ! Double.isNaN( x ) && ! Double.isInfinite( x ) &&
             ! Double.isNaN( y ) && ! Double.isInfinite( y ) ) {
            itemList_.add( new Point( x, y ) );
        }
    }

    /**
     * Returns a new Theory object based on the points which have been
     * added to this factory so far.
     *
     * @param   name   name of the theory
     * @return   new theory
     */
    public Theory createTheory( String name ) {
        XValue.normaliseList( itemList_ );
        int npoint = itemList_.size();
        double[] xs_ = new double[ npoint ];
        double[] ys_ = new double[ npoint ];
        for ( int i = 0; i < npoint; i++ ) {
            Point point = (Point) itemList_.get( i );
            xs_[ i ] = point.x_;
            ys_[ i ] = point.y_;
        }
        return new ArraysTheory( name, xs_, ys_ );
    }

    /**
     * Helper class representing an (x,y) pair.
     */
    private static class Point extends XValue {
        final double x_;
        final double y_;
        Point( double x, double y ) {
            super( x );
            x_ = x;
            y_ = y;
        }
    }
}
