package fit.formats;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Logger;
import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.Theory;
import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.ArraysTheory;
import fit.util.ArraysTheorySet;
import fit.util.StarMetadataTable;
import uk.ac.starlink.table.ArrayColumn;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.ColumnStarTable;
import uk.ac.starlink.table.DefaultValueInfo;
import uk.ac.starlink.table.ValueInfo;
import uk.ac.starlink.util.DataSource;
import uk.ac.starlink.util.DoubleList;

/**
 * Reads Theory objects from a Galaxev output file.
 *
 * @author   Mark Taylor
 * @since    26 Oct 2006
 * @see   <a href="http://www.cida.ve/%7Ebruzual/bc2003">GALAXEV</a>
 */
public class GalaxevTheoryReader implements TheoryReader {

    private static final String OUTFILE_PREFIX = "# Output file name =";
    private static final String INFILE_PREFIX = "# Input file name  =";
    private static final String COLUMNS_PREFIX = "# Column";
    private static final String RECORDS_PREFIX = "# Record";
    private static final String AGES_PREFIX = "# Age (yr)";
    private static final String FLUXES_PREFIX = "# Lambda(A)";
    private static final Logger logger_ = Logger.getLogger( "fit.formats" );
    private static final ValueInfo AGE_INFO =
        new DefaultValueInfo( "Age", Float.class, "Model age" );
    private static final ValueInfo RECORD_INFO =
        new DefaultValueInfo( "Record", Integer.class,
                              "Galaxev record number" );
    static {
        ((DefaultValueInfo) AGE_INFO).setUnitString( "year" );
    }

    public TheorySet readTheories( DataSource datsrc ) throws IOException {
        InputStream in = new BufferedInputStream( datsrc.getInputStream() );
        try {
            return readTheories( in );
        }
        finally {
            in.close();
        }
    }

    /**
     * Reads a Galaxev input file from a stream and returns an array of
     * the Theory objects read from it.
     * The input stream is used raw and not closed, so buffering it
     * upstream would be a good idea.
     *
     * @param   in   input stream
     * @return   array of theories read from <code>in</code>
     */
    private TheorySet readTheories( InputStream in ) throws IOException {
        LineSequence lseq = new LineSequence( in );

        /* Prepare metadata variables. */
        String outfile = null;
        String infile = null;
        String[] icols = null;
        int[] records = null;
        String[] sAges = null;
        float[] ages = null;
        String[] fluxheads = null;

        /* Read header. */
        int iline1 = 0;
        for ( boolean done = false; ! done; ) {
            String line = lseq.nextLine();
            iline1++;
            if ( line == null ) {
                throw new FileFormatException( "No table body " 
                                             + "(not Galaxev format?)" );
            }
            else if ( line.startsWith( OUTFILE_PREFIX ) ) {
                outfile = FormatUtils.readTrail( line, OUTFILE_PREFIX );
            }
            else if ( line.startsWith( INFILE_PREFIX ) ) {
                infile = FormatUtils.readTrail( line, INFILE_PREFIX );
            }
            else if ( line.startsWith( COLUMNS_PREFIX ) ) {
                icols = FormatUtils.readHeadings( line, COLUMNS_PREFIX );
            }
            else if ( line.startsWith( RECORDS_PREFIX ) ) {
                String[] sRecs =
                    FormatUtils.readHeadings( line, RECORDS_PREFIX );
                int nrec = sRecs.length;
                records = new int[ nrec ];
                try {
                    for ( int irec = 0; irec < nrec; irec++ ) {
                        records[ irec ] = Integer.parseInt( sRecs[ irec ] );
                    }
                }
                catch ( NumberFormatException e ) {
                    logger_.warning( "Failed to parse some record ids" );
                    records = null;
                }
            }
            else if ( line.startsWith( AGES_PREFIX ) ) {
                sAges = FormatUtils.readHeadings( line, AGES_PREFIX );
                int nage = sAges.length;
                ages = new float[ nage ];
                try {
                    for ( int iage = 0; iage < nage; iage++ ) {
                        ages[ iage ] = Float.parseFloat( sAges[ iage ] );
                    }
                }
                catch ( NumberFormatException e ) {
                    logger_.warning( "Failed to parse some age values" );
                    ages = null;
                }
            }
            else if ( line.startsWith( FLUXES_PREFIX ) ) {
                fluxheads = FormatUtils.readHeadings( line, FLUXES_PREFIX );
                done = true;
            }
            else if ( line.trim().length() == 0 || line.startsWith( "#" ) ) {
                logger_.warning( "Unexpected galaxev comment line " + iline1 );
            }
            else {
                throw new FileFormatException( "No line \"" + FLUXES_PREFIX + 
                                               "\" - not Galaxev format" );
            }
        }

        /* Validate metadata. */
        int nth = fluxheads.length;
        int ncol = nth + 1;
        if ( sAges == null ) {
            throw new FileFormatException( "No \"" + AGES_PREFIX
                                         + "\" header line" );
        }
        if ( sAges.length != nth ||
             ( ages != null && ages.length != nth ) ||
             ( icols != null && icols.length != nth ) ||
             ( records != null && records.length != nth ) ) {
            logger_.warning( "Row length mismatches?" );
        }

        /* Read data cells. */
        DoubleList[] dataLists = new DoubleList[ ncol ];
        for ( int icol = 0; icol < ncol; icol++ ) {
            dataLists[ icol ] = new DoubleList();
        }
        int nrow = 0;
        double lastX = 0.0;
        String line;
        while ( ( line = lseq.nextLine() ) != null ) {
            iline1++;
            nrow++;
            double[] values = FormatUtils.readNumbers( line, iline1, ncol );
            for ( int icol = 0; icol < ncol; icol++ ) {
                double val = values[ icol ];
                dataLists[ icol ].add( val );
                if ( icol == 0 && nrow > 1 ) {
                    if ( ! ( val > dataLists[ 0 ].get( nrow - 2 ) ) ) {
                        throw new FileFormatException(
                            "Non-increasing wavelengths " + "at line "
                          + iline1 );
                    }
                }
            }
            double x = dataLists[ 0 ].get( 0 );
        }

        /* Convert to array data. */
        double[][] data = new double[ ncol ][];
        for ( int icol = 0; icol < ncol; icol++ ) {
            data[ icol ] = dataLists[ icol ].toDoubleArray();
            assert data[ icol ].length == nrow;
            dataLists[ icol ] = null;
        }

        /* Convert to an array of Theory objects. */
        Theory[] theories = new Theory[ nth ];
        for ( int ith = 0; ith < nth; ith++ ) {
            String name = "Age_" + sAges[ ith ];
            theories[ ith ] = 
                new ArraysTheory( name, data[ 0 ], data[ 1 + ith ] );
        }

        /* Prepare metadata table. */
        ColumnStarTable starTable = ColumnStarTable.makeTableWithRows( nth );
        if ( ages != null ) {
            starTable.addColumn( ArrayColumn
                                .makeColumn( new ColumnInfo( AGE_INFO ),
                                             ages ) );
        }
        if ( records != null ) {
            starTable.addColumn( ArrayColumn
                                .makeColumn( new ColumnInfo( RECORD_INFO ),
                                                             records ) );
        }
        MetadataTable metaTable = starTable.getColumnCount() > 0
                                ? new StarMetadataTable( starTable )
                                : null;

        /* Return. */
        return new ArraysTheorySet( theories, MetadataItem.WAVELENGTH_A,
                                    MetadataItem.FLUX, metaTable );
    }
}
