package fit.formats;

import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.util.MetaStarTable;
import fit.util.FitUtils;
import java.io.IOException;
import java.io.OutputStream;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.EmptyStarTable;
import uk.ac.starlink.table.RandomStarTable;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StarTableWriter;

/**
 * Serialises a {@link fit.framework.TheorySet} object to a tabular format.
 * Each row of the table represents a single {@link fit.framework.Theory}.
 * The first two columns of the format are <code>double[]</code> array
 * types containing (matched) X and Y vectors respectively, and any
 * subsequent columns represent per-theory metadata.
 * This is more or less the "native" format for the fitting program.
 *
 * @author   Mark Taylor
 * @since    19 Jan 2007
 */
public class ModelTableTheoryWriter {

    /**
     * Serialises a given theory set to an output stream.
     * The output stream is not buffered or closed by this routine.
     *
     * @param   tset  theory set to output
     * @param   tWriter   table serializer
     * @param   out    destination stream
     */
    public void write( TheorySet tset, StarTableWriter tWriter,
                       OutputStream out )
            throws IOException {
        tWriter.writeStarTable( toStarTable( tset ), out );
    }

    /**
     * Converts a theory set to a table.
     *
     * @param   tset  theory set
     * @return   tabular representation of <code>tset</code>
     */
    public StarTable toStarTable( TheorySet tset ) {
         return new TheorySetStarTable( tset );
    }

    /**
     * StarTable implementation which represents a TheorySet.
     */
    private static class TheorySetStarTable extends RandomStarTable {

        private final TheorySet tset_;
        private final StarTable metaTable_;
        private final ColumnInfo xInfo_;
        private final ColumnInfo yInfo_;

        /**
         * Constructor.
         *
         * @param   tset  theory set
         */
        public TheorySetStarTable( TheorySet tset ) {
            tset_ = tset;
            MetadataTable meta = tset.getMetadataTable();
            metaTable_ = meta == null ? (StarTable) new EmptyStarTable()
                                      : (StarTable) new MetaStarTable( meta );
            MetadataItem xmeta = tset.getMetadataX();
            MetadataItem ymeta = tset.getMetadataY();
            xInfo_ = new ColumnInfo( xmeta.getName(), double[].class,
                                     "Model X values vector" );
            yInfo_ = new ColumnInfo( ymeta.getName(), double[].class,
                                     "Model Y values vector" );
            if ( xmeta.getUnit() != null ) {
                xInfo_.setUnitString( xmeta.getUnit() );
            }
            if ( ymeta.getUnit() != null ) {
                yInfo_.setUnitString( ymeta.getUnit() );
            }
        }

        public int getColumnCount() {
            return metaTable_.getColumnCount() + 2;
        }

        public long getRowCount() {
            return tset_.getTheoryCount();
        }

        public ColumnInfo getColumnInfo( int icol ) {
            return icol < 2
                 ? new ColumnInfo[] { xInfo_, yInfo_ }[ icol ]
                 : metaTable_.getColumnInfo( icol - 2 );
        }

        public Object getCell( long lrow, int icol ) throws IOException {
            if ( icol >= 2 ) {
                return metaTable_.getCell( lrow, icol - 2 );
            }
            Theory th = tset_.getTheory( checkedLongToInt( lrow ) );
            if ( icol == 0 ) {
                return FitUtils.getXs( th );
            }
            else if ( icol == 1 ) {
                return FitUtils.getYs( th );
            }
            else {
                throw new IllegalArgumentException( "Negative column " + icol );
            }
        }
    }
}
