package fit.formats;

import fit.framework.Interpolator;
import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.ObservationSet;
import fit.util.ArraysObservationSet;
import fit.util.InterpolatorFactory;
import fit.util.StarMetadataTable;
import java.io.IOException;
import java.util.Iterator;
import uk.ac.starlink.table.ColumnPermutedStarTable;
import uk.ac.starlink.table.DescribedValue;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StoragePolicy;
import uk.ac.starlink.table.TableBuilder;
import uk.ac.starlink.table.Tables;
import uk.ac.starlink.table.ValueInfo;
import uk.ac.starlink.util.DataSource;
import uk.ac.starlink.util.IntList;
import uk.ac.starlink.votable.VOTableBuilder;

/**
 * Deserializer for observation sets.
 *
 * @author   Mark Taylor
 * @since    1 Feb 2007
 */
public class ObservationSetReader {

    private final TableBuilder votBuilder_ = new VOTableBuilder();
    private final InterpolatorFactory interpFactory_;

    /**
     * Constructor.
     *
     * @param  interpFactory   interpolator factory - interpolations schemes
     *         are not currently handled by the de/serializer
     */
    public ObservationSetReader( InterpolatorFactory interpFactory ) {
        interpFactory_ = interpFactory;
    }

    /**
     * Reads an observation set from a table in VOTable or FITS format.
     *
     * @param  datsrc   location of data
     * @return  observation set
     */
    public ObservationSet readObservations( DataSource datsrc ) 
            throws IOException {
        return readObservations( datsrc, votBuilder_ );
    }

    /**
     * Reads an observation set from a table using a given handler.
     *
     * @param   datsrc   location of data
     * @param   handler  table input handler - probably only VOTable will work
     * @return  observation set
     */
    private ObservationSet readObservations( DataSource datsrc,
                                             TableBuilder handler )
            throws IOException {
        StarTable table =
            handler.makeStarTable( datsrc, true,
                                   StoragePolicy.getDefaultPolicy() );
        return readObservations( Tables.randomTable( table ) );
    }

    /**
     * Obtains an observation set from its tabular form.
     *
     * @param   table  table
     * @return  equivalent observation set
     */
    private ObservationSet readObservations( StarTable table )
            throws IOException {

        /* Identify columns for Y, Y error and X shift by UCD. */
        IntList colmapList = new IntList( table.getColumnCount() );
        MetadataItem metaY = null;
        int iyCol = -1;
        int iyerrCol = -1;
        int ishiftCol = -1;
        for ( int icol = 0; icol < table.getColumnCount(); icol++ ) {
            ValueInfo info = table.getColumnInfo( icol );
            String ucd = info.getUCD();
            if ( ObservationSetWriter.Y_INFO.getUCD().equals( ucd ) ) {
                iyCol = icol;
                metaY = new MetadataItem( info.getName(),
                                          info.getUnitString() );
            }
            else if ( ObservationSetWriter.YERR_INFO.getUCD().equals( ucd ) ) {
                iyerrCol = icol;
            }
            else if ( ObservationSetWriter.SHIFT_INFO.getUCD().equals( ucd ) ) {
                ishiftCol = icol;
            }
            else {
                colmapList.add( icol );
            }
        }
        if ( iyCol < 0 || iyerrCol < 0 ) {
            throw new FileFormatException( "No y/yerror columns present" );
        }
        assert metaY != null;

        /* Construct the metadata table from the columns which are not 
         * carrying other items. */
        MetadataTable metaTable =
            new StarMetadataTable(
                new ColumnPermutedStarTable( table, colmapList.toIntArray(),
                                             false ) );

        /* Identify parameters representing X and X error using UCDs. */
        double[] xs = null;
        double[] xerrs = null;
        MetadataItem metaX = null;
        for ( Iterator it = table.getParameters().iterator(); it.hasNext(); ) {
            DescribedValue dval = (DescribedValue) it.next();
            ValueInfo info = dval.getInfo();
            String ucd = info.getUCD();
            if ( ObservationSetWriter.X_INFO.getUCD().equals( ucd ) ) {
                xs = (double[]) dval.getValue();
                metaX = new MetadataItem( info.getName(),
                                          info.getUnitString() );
            }
            else if ( ObservationSetWriter.XERR_INFO.getUCD().equals( ucd ) ) {
                xerrs = (double[]) dval.getValue();
            }
        }
        if ( xs == null || xerrs == null ) {
            throw new FileFormatException( "No x/xerror parameters" );
        }
        assert metaX != null;

        /* Populate arrays to construct and return an observation set. */
        int npoint = xs.length;
        int nobs = Tables.checkedLongToInt( table.getRowCount() );
        double[][] ys = new double[ npoint ][ nobs ];
        double[][] yerrs = new double[ npoint ][ nobs ];
        double[] xfactors = ishiftCol >= 0 ? new double[ nobs ] : null;
        for ( int iobs = 0; iobs < nobs; iobs++ ) {
            double[] yVec = (double[]) table.getCell( iobs, iyCol );
            double[] yerrVec = (double[]) table.getCell( iobs, iyerrCol );
            for ( int ip = 0; ip < npoint; ip++ ) {
                ys[ ip ][ iobs ] = yVec[ ip ];
                yerrs[ ip ][ iobs ] = yerrVec[ ip ];
            }
            if ( ishiftCol >= 0 ) {
                xfactors[ iobs ] =
                    ((Number) table.getCell( iobs, ishiftCol )).doubleValue();
            }
        }
        Interpolator[] interps = new Interpolator[ npoint ];
        for ( int ip = 0; ip < npoint; ip++ ) {
            interps[ ip ] =
                interpFactory_.getInterpolator( xs[ ip ], xerrs[ ip ] );
        }
        return new ArraysObservationSet( npoint, nobs, xs, xerrs, interps,
                                         ys, yerrs, xfactors, metaX, metaY,
                                         metaTable );
    }
}
