package fit.formats;

import fit.framework.MetadataItem;
import fit.framework.Theory;
import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.ArraysTheorySet;
import fit.util.FitUtils;
import fit.util.TheoryFactory;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.TableBuilder;
import uk.ac.starlink.table.TableSink;
import uk.ac.starlink.util.DataSource;
import uk.ac.starlink.votable.VOTableBuilder;

/**
 * TheorySet reader which reads data from a table in which the first 
 * column gives X values and each other column gives the Y values for
 * one theory.  Thus there are ncol-1 theories per table.
 *
 * @author   Mark Taylor
 * @since    24 Nov 2006
 */
public class SidewaysTableTheoryReader implements TheoryReader {

    private final TableBuilder handler_;

    /**
     * Constructor.
     *
     * @param   handler  input handler specific to a certain table format
     */
    public SidewaysTableTheoryReader( TableBuilder handler ) {
        handler_ = handler;
    }

    public TheorySet readTheories( DataSource datsrc ) throws IOException {
        InputStream in = new BufferedInputStream( datsrc.getInputStream() );
        try {
            return readTheories( in );
        }
        finally {
            in.close();
        }
    }

    private TheorySet readTheories( InputStream in ) throws IOException {
        SidewaysTheoryReaderSink sink = new SidewaysTheoryReaderSink();
        handler_.streamStarTable( in, sink, null );
        return sink.getTheorySet();
    }

    /**
     * Subclass for sideways tables in VOTable format.
     */
    public static class VOTable extends SidewaysTableTheoryReader {
        public VOTable() {
            super( new VOTableBuilder() );
        }
    }

    /**
     * TableSink implementation which turns a sideways table into a TheorySet.
     */
    private static class SidewaysTheoryReaderSink implements TableSink {

        private TheoryFactory[] thFacts_;
        private String[] names_;
        private TheorySet theorySet_;
        private MetadataItem xmeta_;
        private MetadataItem ymeta_;

        public void acceptMetadata( StarTable table ) {
            int nTh = table.getColumnCount() - 1;
            thFacts_ = new TheoryFactory[ nTh ];
            names_ = new String[ nTh ];
            for ( int ith = 0; ith < nTh; ith++ ) {
                int icol = ith + 1;
                thFacts_[ ith ] = new TheoryFactory();
                names_[ ith ] = "Model_"
                              + table.getColumnInfo( icol ).getName();
            }
            ColumnInfo xInfo = table.getColumnInfo( 0 );
            xmeta_ = new MetadataItem( xInfo.getName(), xInfo.getUnitString() );
            ymeta_ = MetadataItem.Y;
        }

        public void acceptRow( Object[] row ) {
            double x = FitUtils.getNumber( row[ 0 ] );
            int nth = thFacts_.length;
            for ( int ith = 0; ith < nth; ith++ ) {
                int icol = ith + 1;
                thFacts_[ ith ].addPoint( x,
                                          FitUtils.getNumber( row[ icol ] ) );
            }
        }

        public void endRows() {
            int nth = thFacts_.length;
            Theory[] theories = new Theory[ nth ];
            for ( int ith = 0; ith < nth; ith++ ) {
                theories[ ith ] = thFacts_[ ith ].createTheory( names_[ ith ] );
                thFacts_[ ith ] = null;
            }
            thFacts_ = null;
            theorySet_ = new ArraysTheorySet( theories, xmeta_, ymeta_, null );
        }

        /**
         * Returns the TheorySet represented by the table which has been
         * digested by this sink.  Will only return a non-null result if
         * called after {@link #endRows}.
         *
         * @return   theory set
         */
        public TheorySet getTheorySet() {
            return theorySet_;
        }
    }
}
