package fit.formats;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.Theory;
import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.ArraysTheorySet;
import fit.util.FitUtils;
import fit.util.StarMetadataTable;
import fit.util.TheoryFactory;
import uk.ac.starlink.table.ArrayColumn;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.ColumnStarTable;
import uk.ac.starlink.table.DefaultValueInfo;
import uk.ac.starlink.table.ValueInfo;
import uk.ac.starlink.util.FloatList;
import uk.ac.starlink.util.DataSource;

/**
 * Reads theory data from Starburst99 output files.
 *
 * @author   Mark Taylor
 * @since    26 Oct 2006
 * @see <a href="http://www.stsci.edu/science/starburst99/">Starburst99</a>
 */
public class Starburst99TheoryReader implements TheoryReader {

    private static final String DESIG_PREFIX = " MODEL DESIGNATION: ";
    private static final String DATE_PREFIX = " MODEL GENERATED ON ";
    private static final String TITLE_STRING = "COMPUTED SYNTHETIC SPECTRUM";
    private static final String HEADER_REGEX =
        "^ *TIME \\[YR\\] +" +
        "WAVELENGTH \\[A\\] +" +
        "LOG TOTAL +" +
        "LOG STELLAR +" +
        "LOG NEBULAR +" +
        "\\[ERG/SEC/A\\] *";
    private static final ValueInfo TIME_INFO =
        new DefaultValueInfo( "TIME", Float.class, "Model age" );
    static {
       ((DefaultValueInfo) TIME_INFO).setUnitString( "year" );
    }

    private static final Logger logger_ = Logger.getLogger( "fit.formats" );

    public TheorySet readTheories( DataSource datsrc ) throws IOException {
        InputStream in = new BufferedInputStream( datsrc.getInputStream() );
        try {
            return readTheories( in );
        }
        finally {
            in.close();
        }
    }

    private TheorySet readTheories( InputStream in ) throws IOException {
        LineSequence lseq = new LineSequence( in );

        /* Prepare metadata variables. */
        String desig = null;
        String date = null;
        int iline1 = 0;
        for ( boolean done = false; ! done; ) {
            String line = lseq.nextLine();
            iline1++;
            if ( line == null ) {
                throw new FileFormatException( "No table body " 
                                             + "(not Starburst99 format?)" );
            }
            else if ( line.startsWith( DESIG_PREFIX ) ) {
                desig = FormatUtils.readTrail( line, DESIG_PREFIX );
            }
            else if ( line.startsWith( DATE_PREFIX ) ) {
                date = FormatUtils.readTrail( line, DATE_PREFIX );
            }
            else if ( line.indexOf( TITLE_STRING ) >= 0 ) {
                // ignore
            }
            else if ( line.trim().length() == 0 ) {
                // ignore
            }
            else if ( line.matches( HEADER_REGEX ) ) {
                done = true;
            }
            else {
                throw new FileFormatException( "Urecognised header text " 
                                             + "at line " + iline1 );
            }
        }

        /* Read data. */
        TheoryFactory thFact = null;
        float lastTime = Float.NaN;
        List theoryList = new ArrayList();
        FloatList timeList = new FloatList();
        logger_.info( "Starburst99: Flux in erg/s/A will be "
                    + "converted to erg/s/Hz" );
        for ( String line; ( line = lseq.nextLine() ) != null; ) {
            iline1++;
            double[] values = FormatUtils.readNumbers( line, iline1, 5 );
            float time = (float) values[ 0 ];
            double lambdaA = values[ 1 ];
            double logTot = values[ 2 ];
            double logStel = values[ 3 ];
            double logNeb = values[ 4 ];
            if ( time != lastTime ) {
                if ( thFact != null ) {
                    timeList.add( lastTime );
                    theoryList.add( thFact.createTheory( "Age_" + lastTime ) );
                }
                thFact = new TheoryFactory();
                lastTime = time;
            }
            thFact.addPoint( lambdaA,
                             FitUtils.perA2perHz( Math.exp( logTot ),
                                                  lambdaA ) );
        }
        if ( thFact != null ) {
            timeList.add( lastTime );
            theoryList.add( thFact.createTheory( "Age_" + lastTime ) );
        }
        Theory[] theories = (Theory[]) theoryList.toArray( new Theory[ 0 ] );
        float[] times = timeList.toFloatArray();

        /* Prepare metadata table. */
        int nth = theories.length;
        assert nth == times.length;
        ColumnStarTable starTable = ColumnStarTable.makeTableWithRows( nth );
        starTable.addColumn( ArrayColumn
                            .makeColumn( new ColumnInfo( TIME_INFO ), times ) );
        MetadataTable metaTable = starTable.getColumnCount() > 0
                                ? new StarMetadataTable( starTable )
                                : null;

        /* Create and return array. */
        return new ArraysTheorySet( theories, MetadataItem.WAVELENGTH_A,
                                    MetadataItem.FLUX_ERG_S_HZ, metaTable );
    }
}
