package fit.formats;

import fit.framework.MetadataItem;
import fit.framework.Theory;
import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.ArraysTheorySet;
import fit.util.FitUtils;
import fit.util.MapGroupMetadataTable;
import fit.util.TheoryFactory;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.tools.tar.TarEntry;
import org.apache.tools.tar.TarInputStream;
import uk.ac.starlink.table.DescribedValue;
import uk.ac.starlink.table.RowSequence;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StarTableFactory;
import uk.ac.starlink.table.StoragePolicy;
import uk.ac.starlink.table.TableBuilder;
import uk.ac.starlink.table.Tables;
import uk.ac.starlink.util.DataSource;
import uk.ac.starlink.util.MapGroup;
import uk.ac.starlink.votable.VOTableBuilder;

/**
 * TheoryReader for reading tar archives of SVO-style TSAP files.
 * This is a TheoryReader which can read a tar archive
 * of multiple theory VOTables in the style of those returned by the 
 * SVO-LAEFF TSAP service at
 * <a href="http://svo.laeff.esa.es/theory/db2vo/html/"
 *         >http://svo.laeff.esa.es/theory/db2vo/html/"</a>.
 *
 * @author   Mark Taylor
 * @since    15 Jan 2007
 */
public class SvoTarTheoryReader implements TheoryReader {

    private static final StarTableFactory tfact_ = new StarTableFactory( true );
    private static final TableBuilder vbuilder_ = new VOTableBuilder();
    static {
        tfact_.setStoragePolicy( StoragePolicy.PREFER_MEMORY );
    }

    public TheorySet readTheories( DataSource datsrc ) throws IOException {
        InputStream in = new BufferedInputStream( datsrc.getInputStream() );
        try {
            return readTheories( in );
        }
        finally {
            in.close();
        }
    }

    private TheorySet readTheories( InputStream in ) throws IOException {

        /* Check that we have a tar file. */
        if ( ! in.markSupported() ) {
            in = new BufferedInputStream( in );
        }
        assert in.markSupported();
        in.mark( 512 );
        byte[] magic = readFully( in, 500 );
        in.reset();
        if ( ! isTar( magic ) ) {
            throw new FileFormatException( "Not a tar file" );
        }
        
        /* Read each entry in the tar file. */
        TarInputStream tin = new TarInputStream( in );
        MapGroup metaGroup = new MapGroup();
        List theoryList = new ArrayList();
        for ( TarEntry tent; ( tent = tin.getNextEntry() ) != null; ) {
            StarTable table = readVOTable( tin, (int) tent.getSize() );
            TheoryFactory thFact = new TheoryFactory();
            RowSequence rseq = table.getRowSequence();
            try {
                while ( rseq.next() ) {
                    Object[] row = rseq.getRow();

                    /* Get wavelength in Angstrom. */
                    double lambdaA = ((Number) row[ 0 ]).doubleValue();

                    /* Get flux density in erg/cm/cm/s/A. */
                    double fluxPerA = ((Number) row[ 1 ]).doubleValue();

                    /* Calculate flux in Jansky. */
                    double fluxJy = FitUtils.perA2perHz( fluxPerA, lambdaA )
                                  * 1e23;
                    thFact.addPoint( lambdaA, fluxJy );
                }
            }
            finally {
                rseq.close();
            }
            theoryList.add( thFact.createTheory( tent.getName() ) );
            Map meta = new HashMap();
            for ( Iterator paramIt = table.getParameters().iterator();
                  paramIt.hasNext(); ) {
                DescribedValue dval = (DescribedValue) paramIt.next();
                meta.put( dval.getInfo().getName(), dval );
            }
            metaGroup.addMap( meta );
        }

        /* Return the result as a theory set. */
        return new ArraysTheorySet( (Theory[])
                                    theoryList.toArray( new Theory[ 0 ] ),
                                    MetadataItem.WAVELENGTH_A,
                                    MetadataItem.FLUX_JANSKY,
                                    new MapGroupMetadataTable( metaGroup ) );
    }

    /**
     * Reads a VOTable from a stream.
     *
     * @param   in  input stream positioned at start of VOTable text
     * @param   number of bytes constituting votabla text
     * @return   table read from stream
     */
    private static StarTable readVOTable( InputStream in, int size ) 
            throws IOException {
        byte[] buf = readFully( in, size );
        InputStream bin = new ByteArrayInputStream( buf );
        StarTable table = tfact_.makeStarTable( bin, vbuilder_ );
        bin.close();
        return Tables.randomTable( table );
    }

    /**
     * Checks whether a byte buffer represents the magic number of a tar file.
     *
     * @param   buf  magic buffer, at least 262 bytes long
     */
    private static boolean isTar( byte[] magic ) throws IOException {
        return ( magic[ 257 ] == 'u' &&
                 magic[ 258 ] == 's' &&
                 magic[ 259 ] == 't' &&
                 magic[ 260 ] == 'a' &&
                 magic[ 261 ] == 'r' );
    }

    /**
     * Reads a fixed number of bytes from an input stream and return the
     * result in a byte buffer.
     *
     * @param   in  input stream
     * @param   size   number of bytes to read
     * @return   <code>size</code>-element buffer containing file contents
     * @throws  EOFException if there are not enough bytes in <code>in</code>
     */
    private static byte[] readFully( InputStream in, int size )
            throws IOException {
        byte[] buf = new byte[ size ];
        int pos = 0;
        while ( size > 0 ) {
            int nr = in.read( buf, pos, size );
            if ( nr < 0 ) {
                throw new EOFException( "End of file" );
            }
            pos += nr;
            size -= nr;
        }
        assert size == 0;
        return buf;
    }
}
