package fit.framework;

/**
 * Defines an algorithm for comparing two sets of values to each other.
 * These are notionally a theoretical set and an observational set with
 * associated errors, though these semantics are not essential.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public interface FitCalculator {

    /**
     * Calculates the goodness-of-fit score when comparing two sets of values.
     * The returned value is a gooness-of-fit score indicating how well
     * the <code>y2s</code> observations fit the <code>y1s</code>
     * predictions.  Zero means a perfect fit, larger numbers are worse.
     *
     * @param   np   number of points to compare
     * @param   y1s  <code>np</code>-element array of observed values
     * @param   y2s  <code>np</code>-element array of theoretical values
     * @param   errs <code>np</code>-element array of errors on the observed
     *               values
     * @return  non-negative goodness-of-fit score
     */
    double getScore( int np, double[] y1s, double[] y2s, double[] errs );

    /**
     * Calculates a multiplicative scaling factor to be applied to a
     * set of theoretical values which optimises its fit to a set of
     * observed values with supplied errors.
     * The result is the factor by which the <code>y2s</code> values 
     * should be multiplied for best fit to the <code>y1s</code> values.
     *
     * @param   np   number of points to compare
     * @param   y1s  <code>np</code>-element array of observed values
     * @param   y2s  <code>np</code>-element array of theoretical values
     * @param   errs <code>np</code>-element array of errors on the observed
     *               values
     * @return   normalisation factor for best fit
     */
    double getScale( int np, double[] y1s, double[] y2s, double[] errs );
}
