package fit.framework;

/**
 * Describes an item of data.
 * Units may be null.
 * Some constants providing commonly-used descriptions for X and Y values
 * are provided for convenience.
 *
 * @author   Mark Taylor
 * @since    24 Jan 2007
 */
public class MetadataItem {

    private final String name_;
    private final String unit_;

    /** Generic magnitude. */
    public static final MetadataItem MAGNITUDE =
        new MetadataItem( "Magnitude " );

    /** Generic flux.  No units. */
    public static final MetadataItem FLUX =
        new MetadataItem( "Flux" );

    /** Flux in units of Jansky. */
    public static final MetadataItem FLUX_JANSKY =
        new MetadataItem( FLUX.getName(), "Jy" );

    /** Flux in units of erg/sec/&Aring;ngstrom. */
    public static final MetadataItem FLUX_ERG_S_A =
        new MetadataItem( FLUX.getName(), "erg/s/A" );

    /** Flux in units of erg/sec/Hz. */
    public static final MetadataItem FLUX_ERG_S_HZ =
        new MetadataItem( FLUX.getName(), "erg/s/Hz" );

    /** Generic wavelength.  No units. */
    public static final MetadataItem WAVELENGTH =
        new MetadataItem( "Wavelength" );

    /** Wavelength in &Aring;ngstrom. */
    public static final MetadataItem WAVELENGTH_A =
        new MetadataItem( WAVELENGTH.getName(), "A" );

    /** Wavelength in nanometers. */
    public static final MetadataItem WAVELENGTH_NM =
        new MetadataItem( WAVELENGTH.getName(), "nm" );

    /** Wavelength in micrometers. */
    public static final MetadataItem WAVELENGTH_MICRON =
        new MetadataItem( WAVELENGTH.getName(), "micron" );

    /** Generic X coordinate. */
    public static final MetadataItem X = new MetadataItem( "X" );

    /** Generic Y coordinate. */
    public static final MetadataItem Y = new MetadataItem( "Y" );

    /**
     * Constructs a MetadataItem with a given name and unit.
     *
     * @param   name  item name
     * @param   unit  item units
     */
    public MetadataItem( String name, String unit ) {
        name_ = name;
        unit_ = unit;
    }

    /**
     * Constructs a MetadataItem with a given name and no units.
     */
    public MetadataItem( String name ) {
        this( name, null );
    }

    /**
     * Returns the name associated with the value this item describes.
     *
     * @return  name
     */
    public String getName() {
        return name_;
    }

    /**
     * Returns the units associated with the value this item describes.
     *
     * @return  unit string (may be null)
     */
    public String getUnit() {
        return unit_;
    }

    public boolean isCompatible( MetadataItem other ) {
        String unit0 = this.getUnit();
        String unit1 = other.getUnit();
        String name0 = this.getName();
        String name1 = this.getName();
        if ( unit0 != null && unit0.trim().length() > 0 &&
             unit1 != null && unit1.trim().length() > 0 ) {
            return unit0.equalsIgnoreCase( unit1 );
        }
        else {
            if ( unit0 == null && unit1 == null ) {
                return name0.equalsIgnoreCase( name1 );
            }
        }
        return false;
    }

    public String toString() {
        StringBuffer sbuf = new StringBuffer();
        sbuf.append( getName() );
        String unit = getUnit();
        if ( unit != null ) {
            sbuf.append( " / " )
                .append( unit );
        }
        return sbuf.toString();
    }

    public boolean equals( Object other ) {
        if ( other instanceof MetadataItem ) {
            MetadataItem o = (MetadataItem) other;
            return equal( this.getName(), o.getName() )
                && equal( this.getUnit(), o.getUnit() );
        }
        else {
            return false;
        }
    }

    public int hashCode() {
        int code = 23;
        String name = getName();
        String unit = getUnit();
        code = 555 * ( code + ( name == null ? 99 : name.hashCode() ) );
        code = 555 * ( code + ( unit == null ? 77 : unit.hashCode() ) );
        return code;
    }

    /**
     * Determines whether two objects are equal or not.
     * Similar semantics to {@link #equals}, but does not choke on nulls.
     * 
     * @param  o1  first object
     * @param  o2  second object
     * @return  true iff <code>o1</code> is equal to <code>o2</code>
     */
    private static boolean equal( Object o1, Object o2 ) {
        if ( o1 == null && o2 == null ) {
            return true;
        }
        else if ( o1 == null || o2 == null ) {
            return false;
        }
        else {
            return o1.equals( o2 );
        }
    }
}
