package fit.framework;

/**
 * Collects a group of several observations which contain samples taken
 * at a particular set of X values.  
 * While this is not an especially elegant data object, 
 * treating a set of observations which share sampling abscissae
 * may present opportunities for more efficient processing than 
 * addressing a new set of X values for each one.
 * 
 * @author   Mark Tyalor
 * @since    13 Oct 2006
 */
public interface ObservationSet {

    /**
     * Returns the number of observations (sets of points with the same
     * X values).
     *
     * @return   observation count
     */
    int getObsCount();

    /**
     * Returns the number of points in each observation.
     *
     * @return   point count per observation
     */
    int getPointCount();

    /**
     * Returns the X value for a given point which applies to each observation.
     *
     * @param   ip  index of point
     * @return   abscissa for point
     */
    double getX( int ip );

    /**
     * Returns the error on the X value at a given point which applies to
     * each observation.
     *
     * @param   ip   point index
     * @return   error on X value for point
     */
    double getXError( int ip );

    /**
     * Returns an Interpolator object to use for acquiring theoretical
     * Y values corresponding to observational X values a given point.
     *
     * @param   ip   index of point
     * @return   interpolator object suitable for use at <code>ip</code>
     */
    Interpolator getInterpolator( int ip );

    /**
     * Returns the Y value at a given point for a given observation.
     *
     * @param   ip   point index
     * @param   iobs  observation index
     * @return   Y value at <code>ip</code>
     */
    double getY( int ip, int iobs );

    /**
     * Returns the error on the Y value at a given point for a 
     * given observation.
     *
     * @param   ip   point index
     * @param   iobs  observation index
     * @return   error on Y value at <code>ip</code>
     */
    double getYError( int ip, int iobs );

    /**
     * Returns the factor by which the X positions are multiplied relative
     * to theory for a given observation.  This can be used to record
     * redshift of an observation.  In this case, if the X axis represents
     * wavelength, the factor should be <code>(1+z)</code>.
     * For unshifted values, it should of course be unity.
     *
     * @param   iobs  index of observation
     * @return   X shift factor
     */
    double getXFactor( int iobs );

    /**
     * Returns information about the X values of the observations in this set.
     *
     * @return  X values metadata
     */
    MetadataItem getMetadataX();

    /**
     * Returns information about the Y values of the observations in this set.
     *
     * @return  Y values metadata
     */
    MetadataItem getMetadataY();

    /**
     * May return a table which contains additional metadata for the 
     * observations in this set.  If the result is non-null, then
     * the N'th row of the resulting table
     * provides information about the N'th observation in this set.
     *
     * @return   table giving per-observation metadata about this set, or null
     */
    MetadataTable getMetadataTable();
}
