package fit.framework;

import java.io.PrintStream;

/**
 * Prepares a collection of theoretical function descriptions so that
 * they can be compared against observational data sets.
 * The X values at which samples will be taken are required (and are
 * fixed for the life of this object) as well as the definitions 
 * of the theoretical functions themselves.
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class SampleComparator {

    private final Sampling[] samplings_;
    private final TheorySet theorySet_;
    private final PrintStream out_ = System.out;

    /**
     * Constructs a new comparator given an array of X values, 
     * interpolators and theoretical function descriptions.
     *
     * @param   theorySet  array of theoretical function descriptions against
     *          which observations can be compared
     * @param   xs   X values at which functions will be compared
     * @param   interps  interpolators to be used at each of the 
     *          X values (same length as <code>xs</code>)
     */
    public SampleComparator( TheorySet theorySet, double[] xs,
                             Interpolator[] interps ) {
        samplings_ = getSamplings( theorySet, xs, interps );
        theorySet_ = theorySet;
    }

    /**
     * Compares an observation given as an array of Y values and errors
     * against the theories known by this comparator.
     * The sampling definitions (X values and interpolation schemes)
     * of the supplied ordinate/error arrays
     * must be compatible with those of this object.
     * 
     * <p>Optionally (if <code>preScale</code> is true), before the
     * actual fit takes place, each theoretical curve will be multiplied
     * by a scaling factor which is optimised to give the best fit
     * before the actual fitting takes place.  This is appropriate if
     * the theoretical curves have arbitrary or unknown units.
     *
     * @param  ys  array of observation Y values, one for each X point
     * @param  yerrs   array of observation Y errors, one for each X point
     * @param   fitter   object which defines how fitting is done
     * @param   preScale  true iff fitted scaling should be performed on
     *            the theoretical data prior to the actual fits
     * @return  array of comparisons, one for each theory
     */
    public Comparison[] fitData( double[] ys, double[] yerrs,
                                 FitCalculator fitter, boolean preScale ) {
        int ntheory = samplings_.length;
        Comparison[] comparisons = new Comparison[ ntheory ];
        for ( int ith = 0; ith < ntheory; ith++ ) {
            Sampling samp = samplings_[ ith ];
            int npoint = samp.ys_.length;
            double[] sampYs;
            double scale;
            if ( preScale ) { 
                double[] unscaledYs = (double[]) samp.ys_.clone();
                scale = fitter.getScale( npoint, ys, unscaledYs, yerrs );
                for ( int ip = 0; ip < npoint; ip++ ) {
                    unscaledYs[ ip ] *= scale;
                }
                sampYs = unscaledYs;
            }
            else {
                scale = 1.0;
                sampYs = samp.ys_;
            }
            double score = fitter.getScore( npoint, ys, sampYs, yerrs );
            comparisons[ ith ] = new Comparison( samp.theory_, scale, score,
                                                 samp.theorySet_ );
        }
        return comparisons;
    }

    /**
     * Returns this comparator's TheorySet, if it has one.
     *
     * @return  theory set, or null
     */
    public TheorySet getTheorySet() {
        return theorySet_;
    }

    /**
     * Calculates a set of Sampling objects given a set of theories and
     * matched X and interpolator arrays.
     * 
     * @param  theorySet   array of theoretical functions
     * @param  xs   X sample coordinates
     * @param  interps  sample interpolators (same length as <code>xs</code>
     *                  array)
     * @return   array of Sampling objects, one for each Theory
     */
    private static Sampling[] getSamplings( TheorySet theorySet, double[] xs,
                                            Interpolator[] interps ) {
        if ( xs.length != interps.length ) {
            throw new IllegalArgumentException( "Array length mismatch" );
        }
        int npoint = xs.length;
        int ntheory = theorySet.getTheoryCount();
        Sampling[] samplings = new Sampling[ ntheory ];
        for ( int ith = 0; ith < ntheory; ith++ ) {
            Theory theory = theorySet.getTheory( ith );
            double[] ys = new double[ npoint ];
            for ( int ip = 0; ip < npoint; ip++ ) {
                ys[ ip ] = interps[ ip ].getY( theory, xs[ ip ] );
            }
            samplings[ ith ] = new Sampling( theorySet, theory, ys );
        }
        return samplings;
    }

    /**
     * Helper class which defines a set of samples obtained from a
     * theoretical curve.
     */
    private static class Sampling {

        final TheorySet theorySet_;
        final Theory theory_;
        final double[] ys_;

        /**
         * Constructs a new Sampling.
         *
         * @param   theorySet  theory set from which the theory originates
         * @param   theory   theoretical curve
         * @param   ys    samples of <code>theory</code> at each of the
         *                X values for the appropriate comparison
         */
        Sampling( TheorySet theorySet, Theory theory, double[] ys ) {
            theorySet_ = theorySet;
            theory_ = theory;
            ys_ = ys;
        }
    }
}
