package fit.gui;

import fit.framework.Comparison;
import fit.framework.ObservationSet;
import java.awt.Color;
import java.awt.Paint;
import org.jfree.chart.LegendItem;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.DomainOrder;
import org.jfree.data.general.AbstractSeriesDataset;
import org.jfree.data.xy.XYDataset;

/**
 * JFreeChart XYDataset implementation which presents the data representing
 * a set of comparisons of between observational data and theories.
 *
 * @author   Mark Taylor
 * @since    6 Nov 2006
 */
public class ComparisonDataset extends AbstractSeriesDataset
                               implements XYDataset {

    private final ObservationSet obsSet_;
    private final XYLineAndShapeRenderer renderer_;
    private final DomainOrder domainOrder_;
    private Comparison[] comparisons_;
    private double[][] samples_;

    /** Colours used for comparison plotting. */
    private static final Color[] COLORS = new Color[] {
        new Color( 0xf00000 ),
        new Color( 0x0000f0 ),
        Color.green.darker(),
        Color.gray,
        Color.magenta,
        Color.cyan.darker(),
        Color.orange,
        Color.pink,
        Color.yellow,
    };

    /**
     * Constructor.
     *
     * @param  obsSet  observation set against which theories are compared
     */
    public ComparisonDataset( ObservationSet obsSet ) {
        obsSet_ = obsSet;
        comparisons_ = new Comparison[ 0 ];
        renderer_ = new XYLineAndShapeRenderer( false, false ) {
            public LegendItem getLegendItem( int id, int is ) {

                /* Use only one legend item per series, not one for the line
                 * and one for the symbols.  This reduced visual clutter. */
                return is % 2 == 0 ? null
                                 : super.getLegendItem( id, is );
            }
        };

        /* Determine whether the X points from the observation set are
         * monotonic. */
        boolean ascending = true;
        int np = obsSet.getPointCount();
        for ( int ip = 1; ip < np; ip++ ) {
            ascending = ascending
                     && ( obsSet_.getX( ip - 1 ) < obsSet_.getX( ip ) );
        }
        domainOrder_ = ascending ? DomainOrder.ASCENDING : DomainOrder.NONE;
    }

    public DomainOrder getDomainOrder() {
        return domainOrder_;
    }

    /**
     * Configures the comparisons to be represented by this dataset.
     *
     * @param   comparisons  array of Comparison objects to be represented
     */
    public void setComparisons( Comparison[] comparisons ) {
        int nc = comparisons.length;

        /* Calculate Y values for the comparisons; could be done on the fly,
         * but it might be time-consuming. */
        int np = obsSet_.getPointCount();
        double[][] samples = new double[ nc ][ np ];
        for ( int ic = 0; ic < nc; ic++ ) {
            Comparison comp = comparisons[ ic ];
            for ( int ip = 0; ip < np; ip++ ) {
                samples[ ic ][ ip ] =
                    obsSet_.getInterpolator( ip )
                           .getY( comp.getTheory(), obsSet_.getX( ip ) )
                    * comp.getScale();
            }
        }

        /* Update renderer state. */
        Paint[] paints = COLORS;
        for ( int ic = 0; ic < nc; ic++ ) {
            renderer_.setSeriesLinesVisible( ic * 2 + 0, true );
            renderer_.setSeriesShapesVisible( ic * 2 + 0, false );
            renderer_.setSeriesPaint( ic * 2 + 0,
                                      fainter( paints[ ic % paints.length ] ) );
            renderer_.setSeriesLinesVisible( ic * 2 + 1, false );
            renderer_.setSeriesShapesVisible( ic * 2 + 1, true );
        }

        /* Store state. */
        comparisons_ = comparisons;
        samples_ = samples;
        fireDatasetChanged();
    }

    /**
     * Returns the legend item for the given comparison.
     * If that comparison is not represented in this dataset, returns null.
     *
     * @param  comp  comparison
     * @return  legend item
     */
    public LegendItem getLegendItem( Comparison comp ) {
        for ( int ic = 0; ic < comparisons_.length; ic++ ) {
            if ( comparisons_[ ic ] == comp ) {
                return renderer_.getLegendItem( 0, ic * 2 + 1 );
            }
        }
        return null;
    }

    public int getSeriesCount() {
        return comparisons_.length * 2;
    }

    public Comparable getSeriesKey( int is ) {
        return comparisons_[ is / 2 ].getTheory().getName();
    }

    public int getItemCount( int is ) {
        return is % 2 == 0 ? comparisons_[ is / 2 ].getTheory().getCount()
                           : samples_[ is / 2 ].length;
    }

    public double getXValue( int is, int ip ) {
        return is % 2 == 0 ? comparisons_[ is / 2 ].getTheory().getX( ip )
                           : obsSet_.getX( ip );
    }

    public double getYValue( int is, int ip ) {
        Comparison comp = comparisons_[ is / 2 ];
        return is % 2 == 0 ? comp.getTheory().getY( ip ) * comp.getScale()
                           : samples_[ is / 2 ][ ip ];
    }

    public Number getX( int is, int ip ) {
        return new Double( getXValue( is, ip ) );
    }

    public Number getY( int is, int ip ) {
        return new Double( getYValue( is, ip ) );
    }

    /**
     * Returns a renderer suitable for rendering the data from this set.
     * The renderer state will be updated in step with the configuration
     * of this dataset.
     *
     * @return  renderer
     */
    public XYItemRenderer getRenderer() {
        return renderer_;
    }

    /**
     * Returns a paint context which looks like a supplied one but may be
     * a bit fainter.
     *
     * @param  paint  original paint context
     * @return  possibly modified paint context
     */
    private static final Paint fainter( Paint paint ) {
        if ( paint instanceof Color ) {
            Color bg = Color.WHITE;
            Color color = (Color) paint;
            float[] rgba = color.getRGBComponents( new float[ 4 ] );
            float[] bgRgba = bg.getRGBComponents( new float[ 4 ] );
            for ( int i = 0; i < 3; i++ ) {
                rgba[ i ] += 0.6f * ( bgRgba[ i ] - rgba[ i ] );
            }
            return new Color( rgba[ 0 ], rgba[ 1 ], rgba[ 2 ], rgba[ 3 ] );
        }
        else {
            return paint;
        }
    }
}
