package fit.gui;

import fit.framework.Comparison;
import fit.framework.MetadataTable;
import fit.framework.Theory;
import fit.framework.TheorySet;
import java.util.ArrayList;
import java.util.List;

/**
 * TableModel for displaying {@link fit.framework.Comparison} objects
 * (one per row).
 *
 * @author   Mark Taylor
 * @since    31 Oct 2006
 */
public class ComparisonTableModel extends ColumnsTableModel {

    private final List compList_;
    private double minScore_;
    private double maxScore_;

    /**
     * Constructor.
     *
     * @param  theorySet  set from which the theories this model's comparisons
     *         are taken, or null
     */
    public ComparisonTableModel( final TheorySet theorySet ) {
        compList_ = new ArrayList();
        minScore_ = Double.NaN;
        maxScore_ = Double.NaN;

        List colList = new ArrayList();
        colList.add( new ColDef( "Name", String.class ) {
            public Object getValueAt( int irow ) {
                return getComparison( irow ).getTheory().getName();
            }
        } );
        colList.add( new ColDef( "Score", Float.class ) {
            public Object getValueAt( int irow ) {
                double score = getComparison( irow ).getScore();
                return new Float( (float) score );
            }
        } );
        colList.add( new ColDef( "Scaling", Float.class ) {
            public Object getValueAt( int irow ) {
                double scale = getComparison( irow ).getScale();
                return new Float( (float) scale );
            }
        } );
        if ( theorySet != null && theorySet.getMetadataTable() != null ) {
            final MetadataTable metaTable = theorySet.getMetadataTable();
            for ( int im = 0; im < metaTable.getColumnCount(); im++ ) {
                final int imeta = im;
                colList.add( new ColDef( metaTable.getColumnName( imeta ),
                                         metaTable.getColumnClass( imeta ) ) {
                    public Object getValueAt( int irow ) {
                        Theory theory = getComparison( irow ).getTheory();
                        int ith = theorySet.getIndex( theory );
                        return ith >= 0 ? metaTable.getValueAt( irow, imeta )
                                        : null;
                    }
                } );
            }
        }
        setColDefs( (ColDef[]) colList.toArray( new ColDef[ 0 ] ) );
    }

    public int getRowCount() {
        return compList_.size();
    }

    /**
     * Adds a comparison object (row) to this table.
     *
     * @param  comp  comparison to add
     */
    public void addComparison( Comparison comp ) {
        int irow = compList_.size();
        compList_.add( comp );
        double score = comp.getScore();
        if ( ! ( minScore_ <= score ) ) {
            minScore_ = score;
        }
        if ( ! ( maxScore_ >= score ) ) {
            maxScore_ = score;
        }
        fireTableRowsInserted( irow, irow );
    }

    /**
     * Returns the comparison object represented by a given row of this table.
     *
     * @param  irow  row index
     * @return  comparison at row <code>irow</code>
     */
    public Comparison getComparison( int irow ) {
        return (Comparison) compList_.get( irow );
    }

    /**
     * Returns the range of scores represented by comparisons currently 
     * in this table.
     *
     * @return   2-element (minScore,maxScore) array 
     */
    public double[] getScoreRange() {
        return new double[] { minScore_, maxScore_ };
    }
}
