package fit.gui;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.plot.CrosshairState;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRendererState;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.IntervalXYDataset;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.RectangleEdge;

/**
 * Renderer suitable for plotting an <code>IntervalXYDataset</code>,
 * representing the intervals as error bars.
 *
 * @author   Mark Taylor
 * @since    6 Nov 2006
 */
public class ErrorBarRenderer extends XYLineAndShapeRenderer {

    private boolean xBars_;
    private boolean yBars_;

    /**
     * Constructor.
     */
    public ErrorBarRenderer( boolean lines, boolean shapes,
                             boolean xBars, boolean yBars ) {
        super( lines, shapes );
        xBars_ = xBars;
        yBars_ = yBars;
    }

    public void drawItem( Graphics2D g, XYItemRendererState state,
                          Rectangle2D dataArea, PlotRenderingInfo info,
                          XYPlot plot, ValueAxis xAxis, ValueAxis yAxis,
                          XYDataset dataset, int is, int ip,
                          CrosshairState crosshairState, int ipass ) {
        if ( xBars_ || yBars_ ) {
            Graphics2D g2 = (Graphics2D) g.create();
            g2.setPaint( getItemPaint( is, ip ) );
            g2.setStroke( getItemStroke( is, ip ) );
            IntervalXYDataset intervalDataset = (IntervalXYDataset) dataset;
            RectangleEdge xEdge = plot.getDomainAxisEdge();
            RectangleEdge yEdge = plot.getRangeAxisEdge();
            boolean vert = plot.getOrientation() == PlotOrientation.VERTICAL;
            if ( xBars_ ) {
                double y = intervalDataset.getYValue( is, ip ); 
                double xlo = intervalDataset.getStartXValue( is, ip );
                double xhi = intervalDataset.getEndXValue( is, ip );
                double gy = yAxis.valueToJava2D( y, dataArea, yEdge );
                double gxlo = xAxis.valueToJava2D( xlo, dataArea, xEdge );
                double gxhi = xAxis.valueToJava2D( xhi, dataArea, xEdge );
                Line2D line = vert ? new Line2D.Double( gxlo, gy, gxhi, gy )
                                   : new Line2D.Double( gy, gxlo, gy, gxhi );
                if ( dataArea.intersectsLine( line ) ) {
                    g2.draw( line );
                }
            }
            if ( yBars_ ) {
                double x = intervalDataset.getXValue( is, ip );
                double ylo = intervalDataset.getStartYValue( is, ip );
                double yhi = intervalDataset.getEndYValue( is, ip );
                double gx = xAxis.valueToJava2D( x, dataArea, xEdge );
                double gylo = yAxis.valueToJava2D( ylo, dataArea, yEdge );
                double gyhi = yAxis.valueToJava2D( yhi, dataArea, yEdge );
                Line2D line = vert ? new Line2D.Double( gx, gylo, gx, gyhi )
                                   : new Line2D.Double( gylo, gx, gyhi, gx );
                if ( dataArea.intersectsLine( line ) ) {
                    g2.draw( line );
                }
            }
        }
        super.drawItem( g, state, dataArea, info, plot, xAxis, yAxis,
                        dataset, is, ip, crosshairState, ipass );
    }
}
