package fit.gui;

import fit.framework.ObservationSet;
import java.util.Arrays;
import org.jfree.data.DomainOrder;
import org.jfree.data.general.AbstractSeriesDataset;
import org.jfree.data.xy.IntervalXYDataset;
import uk.ac.starlink.util.IntList;

/**
 * XYDataset implementation representing an ObservationSet.
 * This is an adaptor.
 * The {@link #setSelected} method allows selection of which observations
 * from the ObservationSet are represented in this Dataset (default is all).
 *
 * @author   Mark Taylor
 * @since    2 Feb 2007
 */
public class ObservationSetDataset extends AbstractSeriesDataset
                                   implements IntervalXYDataset {

    private final ObservationSet obsSet_;
    private int[] included_;

   /**
    * Constructor.
    *
    * @param   obsSet  observation set
    */
    public ObservationSetDataset( ObservationSet obsSet ) {
        obsSet_ = obsSet;
        boolean[] selected = new boolean[ obsSet.getObsCount() ];
        Arrays.fill( selected, true );
        setSelected( selected );
    }

    /**
     * Returns the ObservationSet for which this Dataset is an adaptor.
     *
     * @return  observation set
     */
    public ObservationSet getObservationSet() {
        return obsSet_;
    }

   /**
    * Sets the observations in the observation set which will be represented
    * by this dataset.
    * Only those observations with a true value in the corresponding
    * element of the <code>selected</code> array will appear.
    * Note that the <code>selected</code> array is used to configure state
    * and then discarded; subsequent modifications of it will not affect
    * behaviour of this object.
    *
    * @param   selected  map of which theories to represent
    */
    public void setSelected( boolean[] selected ) {
        IntList incList = new IntList();
        for ( int i = 0; i < selected.length; i++ ) {
            if ( selected[ i ] ) {
                incList.add( i );
            }
        }
        included_ = incList.toIntArray();
        fireDatasetChanged();
    }

    /**
     * Returns the index into the observation set corresponding to a series
     * index into this dataset.
     *
     * @param  is series index
     * @return  observation index
     */
    public int getObsIndex( int is ) {
        return is < included_.length 
             ? included_[ is ]
             : -1;
    }

    public DomainOrder getDomainOrder() {
        return DomainOrder.NONE;
    }

    public int getSeriesCount() {
        return included_.length;
    }

    public Comparable getSeriesKey( int is ) {
        return "Observation " + ( getObsIndex( is ) + 1 );
    }

    public int getItemCount( int is ) {
        return obsSet_.getPointCount();
    }

    public double getXValue( int is, int ip ) {
        return obsSet_.getX( ip );
    }

    public double getStartXValue( int is, int ip ) {
        return obsSet_.getX( ip ) - obsSet_.getXError( ip );
    }

    public double getEndXValue( int is, int ip ) {
        return obsSet_.getX( ip ) + obsSet_.getXError( ip );
    }

    public double getYValue( int is, int ip ) {
        return obsSet_.getY( ip, getObsIndex( is ) );
    }

    public double getStartYValue( int is, int ip ) {
        int iobs = getObsIndex( is );
        return obsSet_.getY( ip, iobs ) - obsSet_.getYError( ip, iobs );
    }

    public double getEndYValue( int is, int ip ) {
        int iobs = getObsIndex( is );
        return obsSet_.getY( ip, iobs ) + obsSet_.getYError( ip, iobs );
    }

    public Number getX( int is, int ip ) {
        return new Double( getXValue( is, ip ) );
    }
    public Number getStartX( int is, int ip ) {
        return new Double( getStartXValue( is, ip ) );
    }
    public Number getEndX( int is, int ip ) {
        return new Double( getEndXValue( is, ip ) );
    }

    public Number getY( int is, int ip ) {
        return new Double( getYValue( is, ip ) );
    }
    public Number getStartY( int is, int ip ) {
        return new Double( getStartYValue( is, ip ) );
    }
    public Number getEndY( int is, int ip ) {
        return new Double( getEndYValue( is, ip ) );
    }
}
