package fit.gui;

import java.awt.Component;
import java.util.HashMap;
import java.util.Map;
import javax.swing.Icon;
import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import uk.ac.starlink.table.gui.NumericCellRenderer;

/**
 * Customised <code>JTable</code> for use with the fitting display classes.
 * It provides a mechanism to customise the rendering of all cells in 
 * the table which is slightly less painful than installing individual
 * {@link javax.swing.table.TableCellRenderer}s.
 * It also provides numeric renderers which are better (for scientific
 * purposes) than those supplied by a standard JTable.
 *
 * @author   Mark Taylor
 * @since    31 Oct 2006
 */
public class RenderTweakTable extends JTable {

    private final RenderTweaker tweaker_;
    private final Map rendererMap_;
    private static final RenderTweaker NULL_TWEAKER = new RenderTweaker() {
        public void tweakComponent( Component comp, JTable table, Object value,
                                    boolean isSelected, boolean hasFocus,
                                    int irow, int icol ) {
        }
    };

    /**
     * Constructs a table with an optional rendering modification object.
     *
     * @param   tweaker  rendering tweaker - may be null
     */
    public RenderTweakTable( RenderTweaker tweaker ) {
        tweaker_ = tweaker == null ? NULL_TWEAKER
                                   : tweaker;
        rendererMap_ = new HashMap();
    }

    /**
     * Invoked by JTable every time it needs to render a cell.
     */
    public TableCellRenderer getCellRenderer( int irow, int icol ) {
        TableCellRenderer base = super.getCellRenderer( irow, icol );
        if ( ! rendererMap_.containsKey( base ) ) {
            rendererMap_.put( base, new TweakRenderer( base ) );
        }
        return (TableCellRenderer) rendererMap_.get( base );
    }

    /**
     * Provides a custom renderer on request for a given object type.
     */
    public TableCellRenderer getDefaultRenderer( Class clazz ) {
        return Number.class.isAssignableFrom( clazz )
             ? new NumericCellRenderer( clazz )
             : super.getDefaultRenderer( clazz );
    }

    /**
     * Invoked iff <code>getAutoCreateColumnsFromModel()</code> when the
     * model is changed.
     */
    public void createDefaultColumnsFromModel() {

        /* The superclass implementation sets up the right number of 
         * columns, but it does not configure them with renderers. */
        super.createDefaultColumnsFromModel();

        /* Here we configure them with appropriate renderers and set
         * column widths as appropriate. */
        TableModel model = getModel();
        TableColumnModel colModel = getColumnModel();
        int ncol = colModel.getColumnCount();
        for ( int icol = 0; icol < ncol; icol++ ) {
            TableColumn tcol = colModel.getColumn( icol );
            Class clazz = model.getColumnClass( icol );
            TableCellRenderer renderer = getDefaultRenderer( clazz );
            tcol.setCellRenderer( renderer );
            if ( renderer instanceof TweakRenderer ) {
                renderer = ((TweakRenderer) renderer).base_;
            }

            /* If we know the preferred width of the column, set it here as
             * preferred, min and max.  This means that if the table is 
             * resized the known-width (numeric) columns keep the same 
             * size and any others take up the slack.
             * The slight changes between min and max are so that if you
             * try to resize them you can do it a bit - this is likely to
             * be less of a surprise than a column header which is 
             * unresizable. */
            if ( renderer instanceof NumericCellRenderer ) {
                int width = ((NumericCellRenderer) renderer).getCellWidth();
                tcol.setPreferredWidth( width + 10 );
                tcol.setMinWidth( width );
                tcol.setMaxWidth( width + 20 );
            }
            else if ( Icon.class.isAssignableFrom( clazz ) ) {
                tcol.setPreferredWidth( 16 );
                tcol.setMinWidth( 10 );
                tcol.setMaxWidth( 20 );
            }
            else {
                tcol.setMinWidth( 100 );
            }
        }
    }

    /**
     * TableCellRenderer implementation which uses this table's
     * (user-supplied) {@link RenderTweaker} object to modify rendering
     * of each cell.
     */
    private class TweakRenderer implements TableCellRenderer {

        private final TableCellRenderer base_;

        /**
         * Constructor.
         *
         * @param   base  object which performs the basic cell rendering
         */
        TweakRenderer( TableCellRenderer base ) {
            base_ = base;
        }

        public Component getTableCellRendererComponent( JTable table,
                                                        Object value,
                                                        boolean isSelected,
                                                        boolean hasFocus,
                                                        int irow, int icol ) {
            Component comp =
                base_.getTableCellRendererComponent( table, value, isSelected,
                                                     hasFocus, irow, icol );
            tweaker_.tweakComponent( comp, table, value, isSelected, hasFocus,
                                     irow, icol );
            return comp;
        }
    }
}
