package fit.run;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import uk.ac.starlink.table.RowSequence;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StarTableFactory;
import fit.framework.Interpolator;
import fit.framework.MetadataItem;
import fit.framework.ObservationSet;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.util.ArraysTheorySet;
import fit.util.FitUtils;
import fit.util.LinearInterpolator;
import fit.util.MagToFluxObservationSet;
import fit.util.MagToFluxTheory;
import fit.util.TheoryFactory;
import fit.util.TableObservationSetFactory;

/**
 * Example code to do fitting on Anita Richards' data.
 * I think she originally got it from Jeremy Yates.
 * The data themselves don't seem to make much sense (none of the theories
 * are good fits to the data).
 *
 * @author   Mark Taylor
 * @since    13 Oct 2006
 */
public class Anita extends Runner {

    private final StarTableFactory tableFactory_ = new StarTableFactory( true );
    private final static File DATA_DIR = new File( "." );

    public TheorySet readTheories() throws IOException {
        File file = new File( DATA_DIR, "anita-model.vot" );
        StarTable table = tableFactory_.makeStarTable( file.toString(),
                                                       "votable" );
        int nTh = table.getColumnCount() - 1;
        TheoryFactory[] thFacts = new TheoryFactory[ nTh ];
        for ( int ith = 0; ith < nTh; ith++ ) {
            thFacts[ ith ] = new TheoryFactory();
        }
        RowSequence rseq = table.getRowSequence();
        try {
            while ( rseq.next() ) {
                Object[] row = rseq.getRow();
                double x = FitUtils.getNumber( row[ 0 ] );
                for ( int ith = 0; ith < nTh; ith++ ) {
                    int icol = ith + 1;
                    thFacts[ ith ]
                   .addPoint( x, FitUtils.getNumber( row[ icol ] ) );
                }
            }
        }
        finally {
            rseq.close();
        }
        Theory[] theories = new Theory[ nTh ];
        for ( int ith = 0; ith < nTh; ith++ ) {
            int icol = ith + 1;
            String name = "Mass_" + table.getColumnInfo( icol ).getName();
            theories[ ith ] =
                new MagToFluxTheory( thFacts[ ith ].createTheory( name ) );
        }
        return new ArraysTheorySet( theories, MetadataItem.WAVELENGTH_NM,
                                    MetadataItem.FLUX_JANSKY, null );
    }

    public ObservationSet readObservations() throws IOException {
        File file = new File( DATA_DIR, "anita-obs.vot" );
        StarTable table = tableFactory_.makeStarTable( file.toString(),
                                                       "votable" );
        TableObservationSetFactory obsFact =
            new TableObservationSetFactory( table,
                                            MetadataItem.WAVELENGTH_MICRON,
                                            MetadataItem.MAGNITUDE );
        obsFact.setInterpolatorFactory( LinearInterpolator.getFactory() );
        obsFact.definePoint( 0.356, 0., "0.356", "Err_U_T2" );
        obsFact.definePoint( 0.4825, 0., "0.4825", "Err_g_T2" );
        obsFact.definePoint( 0.626, 0., "0.626", "Err_r_T2" );
        obsFact.definePoint( 0.767, 0., "0.767", "Err_i_T2" );
        obsFact.definePoint( 0.910, 0., "0.910", "Err_z_T2" );
        obsFact.definePoint( 1.240, 0., "1.24", "j_msigcom_T1" );
        obsFact.definePoint( 1.660, 0., "1.66", "h_msigcom_T1" );
        obsFact.definePoint( 2.160, 0., "2.16", "k_msigcom_T1" );
        return new MagToFluxObservationSet( obsFact.createObservationSet() );
    }
}
