package fit.run;

import fit.framework.Comparison;
import fit.framework.MetadataTable;
import fit.framework.ObservationSet;
import fit.framework.ResultSink;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.util.FitUtils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import uk.ac.starlink.table.ColumnData;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.ColumnStarTable;
import uk.ac.starlink.table.DefaultValueInfo;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StarTableOutput;
import uk.ac.starlink.table.StarTableWriter;
import uk.ac.starlink.table.Tables;
import uk.ac.starlink.table.ValueInfo;
import uk.ac.starlink.votable.VOTableWriter;

/**
 * ResultSink implementation which assembles a table which is a join 
 * of an input set of observations with a set of comparisons,
 * the join criterion being the best fit.
 * Thus each row consists of cells describing an observation, followed by
 * cells describing the theory that matches it best and information 
 * about that match (such as match score).
 *
 * <p>According to configuration, the table may be written out on
 * <code>close</code>.
 *
 * @author   Mark Taylor
 * @since    13 Nov 2006
 */
public class ResultBestTable implements ResultSink {

    private MetadataTable obsMeta0_;
    private MetadataTable theoryMeta0_;
    private ColumnStarTable resultTable_;
    private List resultList_;

    private static final ValueInfo IOBS_INFO =
        new DefaultValueInfo( "IOBS", Integer.class,
                              "Index of observation in input set" );
    private static final ValueInfo TNAME_INFO =
        new DefaultValueInfo( "TNAME", String.class, "Best fit theory name" );
    private static final ValueInfo TSCORE_INFO =
        new DefaultValueInfo( "TSCORE", Double.class, "Best fit theory score" );
    private static final ValueInfo TSCALE_INFO =
        new DefaultValueInfo( "TSCALE", Double.class,
                              "Best fit theory scaling factor" );

    public synchronized void addResult( ObservationSet obsSet, int iobs, 
                                        Comparison[] comparisons,
                                        TheorySet theorySet ) {

        /* Initialise the output table if this hasn't been done before
         * (i.e. if this is the first result). */
        if ( resultTable_ == null ) {
            initResultTable( obsSet, theorySet );
            resultList_ = new ArrayList();
        }
        assert resultTable_ != null;

        /* Work out the best comparison of those supplied. */
        Comparison bestComp = getBestComparison( comparisons );

        /* Add an entry to the list of results so far. */
        Theory bestTheory = bestComp.getTheory();
        int itheory = theorySet.getIndex( bestTheory );
        resultList_.add( new ResultItem( obsSet.getMetadataTable(),
                                         theorySet.getMetadataTable(),
                                         bestComp, iobs, itheory ) );
    }

    public void close() throws IOException {
        outputTable( resultTable_ );
    }

    /**
     * Disposes of the table which contains the results represented by this
     * object.  The default implementation writes a VOTable to standard output,
     * but this method may be overridden for more specialised purposes.
     *
     * @param   table  table containing results
     */
    protected void outputTable( StarTable table ) throws IOException {
        StarTableWriter writer = new VOTableWriter();
        new StarTableOutput().writeStarTable( table, System.out, writer );
    }

    /**
     * Called once to initialise the results table which holds the useful
     * state of this object.
     *
     * @param   obsSet  observation set which serves as a template for the
     *          observation sets stored in this sink
     * @param   theorySet   theory set which serves as a template for the
     *          theory sets stored in this sink
     */
    private void initResultTable( ObservationSet obsSet, TheorySet theorySet ) {

        /* Check that we're not called more than once. */
        if ( resultTable_ != null ||
             obsMeta0_ != null || theoryMeta0_ != null ) {
            throw new IllegalStateException();
        }

        /* Create a new table object to store results. */
        resultTable_ = new ColumnStarTable() {
            public long getRowCount() {
                return (long) resultList_.size();
            }
        };

        /* Examine the template ObservationSet metadata and add appropriate
         * columns to the results table. */
        obsMeta0_ = obsSet.getMetadataTable();
        if ( obsMeta0_ != null ) {
            StarTable obsStarTable = FitUtils.getStarTable( obsMeta0_ );
            int ncol = obsStarTable.getColumnCount();
            for ( int icol = 0; icol < ncol; icol++ ) {
                ColumnInfo colInfo = obsStarTable.getColumnInfo( icol );
                final int ic = icol;
                resultTable_.addColumn( new ColumnData( colInfo ) {
                    public Object readValue( long lrow ) {
                        ResultItem item = getResult( lrow );
                        MetadataTable meta = item.obsMeta_;
                        return meta == null
                             ? null
                             : meta.getValueAt( item.iobs_, ic );
                    }
                } );
            }
        }
        else {
            resultTable_.addColumn( new ColumnData( IOBS_INFO ) {
                public Object readValue( long lrow ) {
                    int irow = Tables.checkedLongToInt( lrow );
                    return new Integer( irow + 1 );
                }
            } );
        }

        /* Add columns containing information items common to all Comparison
         * objects. */
        resultTable_.addColumn( new ColumnData( TNAME_INFO ) {
            public Object readValue( long lrow ) {
                return getResult( lrow ).comparison_.getTheory().getName();
            }
        } );
        resultTable_.addColumn( new ColumnData( TSCORE_INFO ) {
            public Object readValue( long lrow ) {
                double score = getResult( lrow ).comparison_.getScore();
                return new Double( score );
            }
        } );
        resultTable_.addColumn( new ColumnData( TSCALE_INFO ) {
            public Object readValue( long lrow ) {
                double scale = getResult( lrow ).comparison_.getScale();
                return new Double( scale );
            }
        } );

        /* Examine the template TheorySet metadata and add appropriate columns
         * to the results table. */
        theoryMeta0_ = theorySet.getMetadataTable();
        if ( theoryMeta0_ != null ) {
            StarTable theoryStarTable = FitUtils.getStarTable( theoryMeta0_ );
            int ncol = theoryStarTable.getColumnCount();
            for ( int icol = 0; icol < ncol; icol++ ) {
                ColumnInfo colInfo = theoryStarTable.getColumnInfo( icol );
                final int ic = icol;
                resultTable_.addColumn( new ColumnData( colInfo ) {
                    public Object readValue( long lrow ) {
                        ResultItem item = getResult( lrow );
                        MetadataTable meta = item.theoryMeta_;
                        return meta == null
                             ? null
                             : meta.getValueAt( item.itheory_, ic );
                    }
                } );
            }
        }
    }

    /**
     * Returns the table which contains the best match information.
     *
     * @return  result table
     */
    public StarTable getResultTable() {
        return resultTable_;
    }
  
    /**
     * Returns the ResultItem corresponding to the <code>lrow</code>'th
     * invocation of {@link #addResult}.
     *
     * @param  lrow  row index
     * @return   result item at <code>lrow</code>
     */
    private ResultItem getResult( long lrow ) {
        return (ResultItem) resultList_.get( Tables.checkedLongToInt( lrow ) );
    }

    /**
     * Returns the best comparison of an array of them.  This is the
     * one with the lowest fitting score.
     *
     * @param  comps   array of candidate comparisons
     * @return   lowest-scored comparison
     */
    private static Comparison getBestComparison( Comparison[] comps ) {
        int nc = comps.length;
        Comparison best = null;
        for ( int ic = 0; ic < nc; ic++ ) {
            Comparison comp = comps[ ic ];
            double score = comp.getScore();
            if ( score >= 0 ) {
                if ( best == null || score < best.getScore() ) {
                    best = comp;
                }
            }
        }
        return best;
    }

    /**
     * Helper class which aggregates information about a single row of
     * the result table.
     */
    private class ResultItem {

        final MetadataTable obsMeta_;
        final MetadataTable theoryMeta_;
        final Comparison comparison_;
        final int iobs_;
        final int itheory_;
      
        /**
         * Constructor.
         *
         * @param   obsMeta   observation metadata table
         * @param   theoryMeta  theory metadata table
         * @param   comparison  selected (best) comparison for row
         * @param   iobs   index of observation into observation metadata table
         * @param   itheory   index of theory into theory metadata table
         */
        ResultItem( MetadataTable obsMeta, MetadataTable theoryMeta,
                    Comparison comparison, int iobs, int itheory ) {
            obsMeta_ = FitUtils.isCompatible( obsMeta, obsMeta0_ )
                     ? obsMeta : null;
            theoryMeta_ = FitUtils.isCompatible( theoryMeta, theoryMeta0_ )
                        ? theoryMeta : null;
            comparison_ = comparison;
            iobs_ = iobs;
            itheory_ = itheory;
        }
    }
}
