package fit.run;

import java.io.IOException;
import java.io.OutputStream;
import fit.framework.Comparison;
import fit.framework.ObservationSet;
import fit.framework.ResultSink;
import fit.framework.TheorySet;

/**
 * ResultSink implementation which writes human-readable results in a 
 * not-very-sophisticated way to an output stream.
 *
 * @author   Mark Taylor
 * @since    27 Oct 2006
 */
public class ResultPrinter implements ResultSink {

    private final OutputStream out_;
    private final byte[] lineSep_;

    /**
     * Constructs a printer that dumps results to standard output.
     */
    public ResultPrinter() {
        this( System.out );
    }

    /**
     * Constructs a printer that dumps results to a given output stream.
     *
     * @param   out   destination stream
     */
    public ResultPrinter( OutputStream out ) {
        out_ = out;
        String sep;
        try {
            sep = System.getProperty( "line.separator", "\n" );
        }
        catch ( SecurityException e ) {
            sep = "\n";
        }
        lineSep_ = sep.getBytes();
    }

    public void addResult( ObservationSet obsSet, int iobs,
                           Comparison[] comparisons, TheorySet theorySet )
            throws IOException {
        println( "" );
        println( "Observation: " + ( iobs + 1 ) );
        Comparison best = comparisons[ 0 ];
        for ( int ic = 0; ic < comparisons.length; ic++ ) {
            Comparison comp = comparisons[ ic ];
            if ( comp.getScore() < best.getScore() ) {
                best = comp;
            }
            println( comp.toString() );
        }
        println( "      Best: " + best.getTheory().getName() );
    }

    public void close() throws IOException {
        out_.close();
    }

    /**
     * Writes a line of text with trailing newline to this printers output
     * stream.
     *
     * @param  line  text (without newline)
     */
    private void println( String line ) throws IOException {
        out_.write( line.getBytes() );
        out_.write( lineSep_ );
    }
}
