package fit.run;

import fit.framework.Comparison;
import fit.framework.ObservationSet;
import fit.framework.ResultSink;
import fit.framework.TheorySet;
import fit.util.FitUtils;
import java.io.IOException;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.DefaultValueInfo;
import uk.ac.starlink.table.DescribedValue;
import uk.ac.starlink.table.RandomStarTable;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.StarTableOutput;
import uk.ac.starlink.table.ValueInfo;

/**
 * ResultSink which writes out each group of comparisons to a table.
 *
 * @author   Mark Taylor
 * @since    27 Oct 2006
 */
public class ResultTabulator implements ResultSink {

    private final String fileBase_;
    private final String fileExten_;
    private final String ofmt_;
    private final StarTableOutput sto_;

    private static final Logger logger_ = Logger.getLogger( "fit.run" );

    /**
     * Constructs a tabulator which writes VOTables to the current directory.
     */
    public ResultTabulator() {
        this( "out-", ".vot", "votable" );
    }

    /**
     * Constructs a tabulator with configurable output characteristics.
     *
     * @param   fileBase   prefix part of filename for each output table file
     * @param   fileExten  postfix part of filename for each output table file
     * @param   ofmt   output table format 
     *          (see <code>StarTableOutput.writeStarTable</code>)
     */
    public ResultTabulator( String fileBase, String fileExten, String ofmt ) {
        fileBase_ = fileBase;
        fileExten_ = fileExten;
        ofmt_ = ofmt;
        sto_ = new StarTableOutput();
    }

    public void addResult( ObservationSet obsSet, int iobs,
                           Comparison[] comparisons, TheorySet theorySet )
            throws IOException {
        comparisons = (Comparison[]) comparisons.clone();
        Arrays.sort( comparisons );
        String tname = fileBase_ + ( iobs + 1 ) + fileExten_;
        sto_.writeStarTable( getResultTable( obsSet, iobs, comparisons,
                                             theorySet ),
                             tname, ofmt_ );
    }

    public void close() {
    }

    /**
     * Returns a table describing the results of a fit of one observation
     * to a set of theories.
     *
     * @param   obsSet   observation set
     * @param   iobs     index of the observation within <code>obsSet</code>
     * @param   comparisons  results of fitting
     * @return   table
     */
    public static StarTable getResultTable( ObservationSet obsSet, int iobs,
                                            Comparison[] comparisons,
                                            TheorySet theorySet ) {
        return new ResultTable( obsSet, iobs, comparisons, theorySet );
    }

    /**
     * StarTable implementation summarising the result of matching an
     * observation to several theories.
     */
    private static class ResultTable extends RandomStarTable {

        private final ValueInfo X_INFO =
            new DefaultValueInfo( "X", Double.class, "X value" );
        private final ValueInfo Y_INFO =
            new DefaultValueInfo( "Y", Double.class, "Y value" );
        private final ValueInfo YERR_INFO =
            new DefaultValueInfo( "YERR", Double.class, "Y error" );
        private final ColumnInfo[] preCols_ = {
            new ColumnInfo( X_INFO ),
            new ColumnInfo( Y_INFO ),
            new ColumnInfo( YERR_INFO ),
        };

        private final ObservationSet obsSet_;
        private final int iobs_;
        private final Comparison[] comparisons_;
        private final TheorySet theorySet_;

        /**
         * Constructor.
         *
         * @param   obsSet   observation set
         * @param   iobs     index of the observation within <code>obsSet</code>
         * @param   comparisons  results of fitting
         * @param   theorySet  theories referenced in <code>comparisons</code>
         *                     (may be null)
         */
        ResultTable( ObservationSet obsSet, int iobs,
                     Comparison[] comparisons, TheorySet theorySet ) {
            obsSet_ = obsSet;
            iobs_ = iobs;
            comparisons_ = comparisons;
            theorySet_ = theorySet;
        }

        public long getRowCount() {
            return (long) obsSet_.getPointCount();
        }

        public int getColumnCount() {
            return comparisons_.length + preCols_.length;
        }

        public ColumnInfo getColumnInfo( int icol ) {
            int icomp = icol - preCols_.length;
            if ( icomp < 0 ) {
                return preCols_[ icol ];
            }
            else {
                Comparison comp = comparisons_[ icomp ];
                String descrip = 
                    "Y * " + (float) comp.getScale() + " for theory " +
                    comp.getTheory().getName() + "; score=" +
                    (float) comp.getScore();
                ColumnInfo colInfo = new ColumnInfo( comp.getTheory().getName(),
                                                     Double.class, descrip );
                if ( theorySet_ != null &&
                     theorySet_.getMetadataTable() != null ) {
                    int ith = theorySet_.getIndex( comp.getTheory() );
                    if ( ith > 0 ) {
                        StarTable metaTable = FitUtils
                            .getStarTable( theorySet_.getMetadataTable() );
                        int nmeta = metaTable.getColumnCount();
                        for ( int imeta = 0; imeta < nmeta; imeta++ ) {
                            ValueInfo info = metaTable.getColumnInfo( imeta );
                            try {
                                Object value = metaTable.getCell( ith, imeta );
                                DescribedValue dval =
                                    new DescribedValue( info, value );
                                colInfo.setAuxDatum( dval );
                            }
                            catch ( IOException e ) {
                                logger_.log( Level.WARNING,
                                             "Metadata access failure", e );
                            }
                        }
                    }
                }
                return colInfo;
            }
        }

        public Object getCell( long lrow, int icol ) {
            return new Double( getCellValue( checkedLongToInt( lrow ), icol ) );
        }

        private double getCellValue( int irow, int icol ) {
            int icomp = icol - preCols_.length;
            if ( icomp < 0 ) {
                switch ( icol ) {
                    case 0: return obsSet_.getX( irow );
                    case 1: return obsSet_.getY( irow, iobs_ );
                    case 2: return obsSet_.getYError( irow, iobs_ );
                    default: throw new AssertionError();
                }
            }
            else {
                Comparison comp = comparisons_[ icomp ];
                return comp.getScale()
                     * obsSet_.getInterpolator( irow )
                              .getY( comp.getTheory(), obsSet_.getX( irow ) );
            }
        }
    }
}
