package fit.run;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import uk.ac.starlink.table.StarTableFactory;
import fit.framework.FitCalculator;
import fit.framework.ObservationSet;
import fit.framework.ResultSink;
import fit.framework.Theory;
import fit.framework.TheoryReader;
import fit.framework.TheorySet;
import fit.util.ChiSquared;
import fit.util.FitUtils;
import fit.util.Poisson;

/**
 * Harness for running fitting data tests.
 * As well as containing a main method for invocation,
 * this also serves as an abstract superclass for test run classes
 * which know how to read data (Theory and ObservationSet objects)
 * from files in some way.
 *
 * @author   Mark Taylor
 * @since    23 Oct 2006
 */
public abstract class Runner {

    protected final static File DATA_DIR = new File( "." );
    protected final StarTableFactory tableFactory_ =
        new StarTableFactory( true );
    private static final Logger logger = Logger.getLogger( "fit.run" );

    /**
     * Acquires an ObservationSet for the fitting.
     *
     * @return  observation data
     */
    protected abstract ObservationSet readObservations() throws IOException;

    /**
     * Acquires an array of Theories for the fitting.
     *
     * @return   theoretical function descriptions
     */
    protected abstract TheorySet readTheories() throws IOException;

    /**
     * Instance method invoked by {@link #main}.
     *
     * @return   exit status
     */
    public int run( String[] args ) throws IOException {
        List argList = new ArrayList( Arrays.asList( args ) );

        /* Process arguments to discover what to do with the results. */
        Map sinkMap = getSinkMap();
        StringBuffer ubuf = new StringBuffer( "Usage: Runner <classname>" );
        String[] sinkFlags = (String[]) new ArrayList( sinkMap.keySet() )
                                       .toArray( new String[ 0 ] );
        Arrays.sort( sinkFlags );
        for ( int is = 0; is < sinkFlags.length; is++ ) {
            ubuf.append( " [" )
                .append( sinkFlags[ is ] )
                .append( "]" );
        }
        List sinkList = new ArrayList();
        for ( Iterator it = argList.iterator(); it.hasNext(); ) {
            String arg = (String) it.next();
            if ( sinkMap.containsKey( arg ) ) {
                it.remove();
                ResultSink sink;
                try {
                    Class clazz = Class.forName( (String) sinkMap.get( arg ) );
                    sink = (ResultSink) clazz.newInstance();
                }
                catch ( Throwable e ) {
                    System.err.println( "No " + arg + " output" );
                    e.printStackTrace( System.err );
                    return 1;
                }
                sinkList.add( sink );
            }
        }

        /* Report error if there were unused command-line arguments. */
        String usage = ubuf.toString();
        if ( argList.size() != 0 ) {
            System.err.println( usage );
            return 1;
        }

        /* Warn if no specification for output has been made. */
        ResultSink[] sinks =
            (ResultSink[]) sinkList.toArray( new ResultSink[ 0 ] );
        if ( sinks.length == 0 ) {
            logger.warning( "No result sinks - results will be discarded" );
        }

        /* Do the work. */
        run( readTheories(), readObservations(), sinks );
        return 0;
    }

    /**
     * Given the input theories and observations, performs matches and 
     * outputs the results in some way (currently crude).
     *
     * @param   theorySet   array of theoretical functions to fit against
     * @param   obsSet   set of observational data to be fitted
     * @param   sinks    destinations for calculated results
     */
    public void run( TheorySet theorySet, ObservationSet obsSet,
                     ResultSink[] sinks )
            throws IOException {
        boolean preScale = true;
        FitCalculator fitCalc = new ChiSquared( preScale ? 1 : 0 );
        FitUtils.doFitting( theorySet, obsSet, preScale, fitCalc, sinks );
        for ( int is = 0; is < sinks.length; is++ ) {
            sinks[ is ].close();
        }
    } 

    /**
     * Returns a flag name -> classname map for ResultSink classes.
     *
     * @return  map of ResultSink class names
     */
    public static Map getSinkMap() {
        Map map = new HashMap();
        map.put( "-summary", ResultSummary.class.getName() );
        map.put( "-stdout", ResultPrinter.class.getName() );
        map.put( "-votout", ResultTabulator.class.getName() );
        map.put( "-gui", "fit.run.ResultDisplay" );
        map.put( "-table", ResultBestTable.class.getName() );
        return map;
    }

    /**
     * Main method.
     * <code>args[0]</code> is the name of a concrete subclass of Runner.
     */
    public static void main( String[] args ) throws IOException {
        List argList = new ArrayList( Arrays.asList( args ) );
        Runner runner;
        try { 
            runner = (Runner) Class.forName( args[ 0 ] ).newInstance();
            argList.remove( 0 );
        }
        catch ( Throwable e ) {
            e.printStackTrace();
            System.err.println( "Usage: Runner <runner-class-name> [args]" );
            System.exit( 1 );
            throw new AssertionError();
        }
        runner.run( (String[]) argList.toArray( new String[ 0 ] ) );
    }
}
