package fit.task;

import fit.formats.ObservationSetReader;
import fit.framework.ObservationSet;
import fit.util.InterpolatorFactory;
import fit.util.LinearInterpolator;
import fit.util.SquareSmoother;
import java.io.IOException;
import uk.ac.starlink.task.ChoiceParameter;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.ParameterValueException;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.util.DataSource;

/**
 * Parameter which reads an ObservationSet in the special serialised format.
 *
 * @author   Mark Taylor
 * @since    2 Feb 2007
 */
public class ObservationSetParameter extends Parameter {

    private final ChoiceParameter interpParam_;
    private ObservationSet obsSet_;

    /**
     * Constructor.
     *
     * @param  name   parameter name
     */
    public ObservationSetParameter( String name ) {
        super( name );
        setUsage( "<obs-file>" );
        setPrompt( "File containing observation data" );

        final String squareOpt = "square";
        final String pointOpt = "point";
        interpParam_ = new ChoiceParameter( "smoother" );
        interpParam_.setPrompt( "Theory smoothing kernel" );
        interpParam_.addOption( SquareSmoother.getFactory(), squareOpt );
        interpParam_.addOption( LinearInterpolator.getFactory(), pointOpt );
        interpParam_.setDefault( interpParam_.getOptionNames()[ 0 ] );
        interpParam_.setDescription( new String[] {
            "<p>Describes how model curves will be sampled over the",
            "width of each observation point for the observations",
            "listed in the <code>" + name + "</code> parameter.",
            "The options are currently",
            "<ul>",
            "<li><code>" + squareOpt + "</code>: "
                 + "the model curve will be uniformly ",
                   "averaged over the width of the observation</li>",
            "<li><code>" + pointOpt + "</code>: "
                 + "a linear interpolation of the model curve",
                   "will be taken at the central point of the observation</li>",
            "</ul>",
            "</p>",
        } );
    }

    /**
     * Returns the associated parameter which selects the interpolator 
     * factory to use.
     *
     * @return   interpolator factory parameter
     */
    public Parameter getInterpolatorParameter() {
        return interpParam_;
    }

    public void setValueFromString( Environment env, String sval )
            throws TaskException {
        InterpolatorFactory interpFact =
            (InterpolatorFactory) interpParam_.objectValue( env );
        try {
            DataSource datsrc = DataSource.makeDataSource( sval );
            obsSet_ = new ObservationSetReader( interpFact )
                     .readObservations( datsrc );
        }
        catch ( IOException e ) {
            throw new ParameterValueException( this, e );
        }
        super.setValueFromString( env, sval );
    }

    /**
     * Returns the value of this parameter as an observation set.
     *
     * @param  env  execution environment
     * @return  observation set
     */
    public ObservationSet observationSetValue( Environment env )
            throws TaskException {
        checkGotValue( env );
        return obsSet_;
    }
}
