package fit.task;

import fit.framework.ObservationSet;
import fit.gui.ObservationSetDisplay;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.Executable;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.SingleTaskInvoker;
import uk.ac.starlink.task.Task;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.task.TerminalEnvironment;

/**
 * Task which does graphical display of an input ObservationSet only. 
 *
 * @author   Mark Taylor
 * @since    2 Feb 2007
 */
public class PlotObservationTask implements Task {

    private final ObservationSetParameter obsParam_;

    /**
     * Constructor.
     */
    public PlotObservationTask() {
        obsParam_ = new ObservationSetParameter( "in" );
        obsParam_.setPrompt( "Location of observation set file" );
        obsParam_.setDescription( new String[] {
            "<p>File containing the observations to be plotted.",
            "This must be in <code>yobs</code> format.",
            "</p>",
        } );
    }

    public String getPurpose() {
        return "Plots observation data";
    }

    public Parameter[] getParameters() {
        return new Parameter[] {
            obsParam_,
        };
    }

    public Executable createExecutable( Environment env ) throws TaskException {
        final ObservationSet obsSet = obsParam_.observationSetValue( env );
        return new Executable() {
            public void execute() {
                SwingUtilities.invokeLater( new Runnable() {
                    public void run() {
                        display( obsSet );
                    }
                } );
            }
        };
    }

    /**
     * Does display of a given ObservationSet.  Must be called from the
     * event dispatch thread.
     *
     * @param  obsSet  observation set to display
     */
    private static void display( ObservationSet obsSet ) {
        JFrame frame = new JFrame( "Observation data" );
        frame.getContentPane().add( new ObservationSetDisplay( obsSet ) );
        frame.setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
        frame.pack();
        frame.setVisible( true );
    }

    /**
     * Invokes this task.  Try "-help".
     */
    public static void main( String[] args ) {
        TerminalEnvironment.NUM_TRIES = 1;
        new SingleTaskInvoker( new PlotObservationTask(), "plotobs" )
           .invoke( args );
    }
}
