package fit.task;

import fit.formats.ObservationSetWriter;
import fit.framework.Interpolator;
import fit.framework.MetadataItem;
import fit.framework.ObservationSet;
import fit.util.ArraysObservationSet;
import fit.util.FitUtils;
import fit.util.InterpolatorFactory;
import fit.util.SquareSmoother;
import fit.util.TableObservationSetFactory;
import gnu.jel.CompilationException;
import gnu.jel.CompiledExpression;
import gnu.jel.Evaluator;
import gnu.jel.Library;
import java.io.IOException;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.Tables;
import uk.ac.starlink.task.Environment;
import uk.ac.starlink.task.Executable;
import uk.ac.starlink.task.OutputStreamParameter;
import uk.ac.starlink.task.Parameter;
import uk.ac.starlink.task.ParameterValueException;
import uk.ac.starlink.task.Task;
import uk.ac.starlink.task.TaskException;
import uk.ac.starlink.ttools.JELRowReader;
import uk.ac.starlink.ttools.JELUtils;
import uk.ac.starlink.ttools.RandomJELRowReader;
import uk.ac.starlink.ttools.task.InputTableParameter;
import uk.ac.starlink.util.Destination;

/**
 * Task to turn a table into an ObservationSet under control of a key file.
 *
 * @author   Mark Taylor
 * @since    6 Feb 2007
 */
public class TableObservationSetTask implements Task {

    private final InputTableParameter obsTableParam_;
    private final ObservationKeyParameter obsKeyParam_;
    private final OutputStreamParameter outParam_;
    private final Parameter zParam_;
    private final Parameter xParam_;
    private final Parameter yParam_;
    private final Parameter xnameParam_;
    private final Parameter xunitParam_;
    private final Parameter ynameParam_;
    private final Parameter yunitParam_;

    /**
     * Constructor.
     */
    public TableObservationSetTask() {
        obsTableParam_ = new InputTableParameter( "in" );
        obsKeyParam_ = new ObservationKeyParameter( "key" );
        obsTableParam_.setPrompt( "Table containing observation data" );
        Parameter formatParam = obsTableParam_.getFormatParameter();
        formatParam.setName( "ifmt" );
        obsTableParam_.setDescription( new String[] {
            "<p>Location of the table which contains the input observation",
            "data.",
            "The columns of this table must contain the Y (flux)",
            "values and errors at different band passes.",
            "Which column means what is specified by the",
            "<code>" + obsKeyParam_.getName() + "</code> parameter.",
            "The table located by this parameter may be in any of the",
            "formats supported by STIL, as defined by the",
            "<code>" + formatParam.getName() + "</code> parameter.",
            "</p>"
        } );
        formatParam.setPrompt( "Format of table " + obsTableParam_.getName() );
        formatParam.setDescription( new String[] {
            "<p>Gives the format of the table located by the",
            "<code>" + obsTableParam_.getName() + "</code> parameter",
            "(<code>votable</code>, <code>fits</code>, <code>csv</code> etc).",
            "</p>",
        } );

        obsKeyParam_.setDescription( new String[] {
            "<p>Location of file containing the four-column <em>key</em>",
            "file that describes which columns of the input table give",
            "which information.",
            "See the command description for more detail.",
            "</p>",
        } );

        outParam_ = new OutputStreamParameter( "out" );
        outParam_.setUsage( "<obs-file>" );
        outParam_.setPrompt( "Name of output observation file" );
        outParam_.setDescription( new String[] {
            "<p>Name of the file to which the output <code>yobs</code>",
            "file will be written.",
            "</p>",
        } );

        zParam_ = new Parameter( "redshift" );
        zParam_.setUsage( "<expr>" );
        zParam_.setPrompt( "Expression (e.g. col name) for "
                         + "observation redshifts" );
        zParam_.setNullPermitted( true );
        zParam_.setDescription( new String[] {
            "<p>May give an expression for redshift of the input observations",
            "in terms of quantities in the input table.",
            "If there is one, this will normally be expressed just as the",
            "name of the column in the input table which gives redshift.",
            "However, it can be an algebraic expression combining column",
            "names from the input table - see <ref id='jel'/>.",
            "A redshift should only be given if the X values are wavelength",
            "(not, for instance, if they are frequencies)",
            "since otherwise the fitting program will not treat them",
            "properly.",
            "</p>",
        } );

        xParam_ = new Parameter( "x" );
        xParam_.setUsage( "<expr>" );
        xParam_.setPrompt( "Transformation expression for X coordinates" );
        xParam_.setDefault( "x" );
        xParam_.setDescription( new String[] {
            "<p>Formula for the X (wavelength) values in the output",
            "observation file in terms of the X values in the key file.",
            "The default value of \"<code>x</code>\" simply makes the one",
            "equal to the other, however it is possible to use algebraic",
            "expressions here in terms of <code>x</code>, for instance",
            "in order to modify the scaling (change units).",
            "Normal arithmetic operators as well as some special functions",
            "may be used - see <ref id='jel'/>.",
            "</p>",
        } );

        yParam_ = new Parameter( "y" );
        yParam_.setUsage( "<expr>" );
        yParam_.setPrompt( "Transformation expression for Y coordinates" );
        yParam_.setDefault( "y" );
        yParam_.setDescription( new String[] {
            "<p>Formula for the Y (flux) values in the output",
            "observation file in terms of the Y values in the input table.",
            "the default value of \"<code>y</code>\" simpley makes the one",
            "equal to the other, however it is possible to use algebraic",
            "expressions here in terms of <code>y</code>, <code>x</code>",
            "(the X values from the input key file), and any of the",
            "values in the input table referred to by their column name.",
            "Normal arithmetic operators as well as some special functions",
            "such as <code>abToJansky()</code> may be used -",
            "see <ref id='jel'/>",
            "</p>",
        } );

        xnameParam_ = new Parameter( "xname" );
        xnameParam_.setPrompt( "Name for output X values" );
        xnameParam_.setDefault( "X" );
        xnameParam_.setDescription( new String[] {
            "<p>Gives a name for the output X axis, such as \"Wavelength\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );

        xunitParam_ = new Parameter( "xunit" );
        xunitParam_.setPrompt( "Unit description for output X values" );
        xunitParam_.setNullPermitted( true );
        xunitParam_.setDescription( new String[] {
            "<p>Gives a unit for the output X axis, such as \"Angstrom\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );

        ynameParam_ = new Parameter( "yname" );
        ynameParam_.setPrompt( "Name for output Y values" );
        ynameParam_.setDefault( "Y" );
        ynameParam_.setDescription( new String[] {
            "<p>Gives a name for the output Y axis, such as \"Flux\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );

        yunitParam_ = new Parameter( "yunit" );
        yunitParam_.setPrompt( "Unit description for output Y values" );
        yunitParam_.setNullPermitted( true );
        yunitParam_.setDescription( new String[] {
            "<p>Gives a unit for the output Y axis, such as \"Jansky\".",
            "This does not affect processing, but may be used to annotate",
            "the displayed or output results, so it is a good idea to supply",
            "a value if known to reduce confusion.",
            "</p>",
        } );
    }

    public String getPurpose() {
        return "Prepares an observation file from a table of observations";
    }

    public Parameter[] getParameters() {
        return new Parameter[] {
            obsTableParam_,
            obsTableParam_.getFormatParameter(),
            obsKeyParam_,
            outParam_,
            zParam_,
            xParam_,
            yParam_,
            xnameParam_,
            xunitParam_,
            ynameParam_,
            yunitParam_,
        };
    }

    public Executable createExecutable( Environment env ) throws TaskException {
        StarTable inTable = obsTableParam_.tableValue( env );
        try {
            inTable = Tables.randomTable( inTable );
        }
        catch ( IOException e ) {
            throw new TaskException( "Trouble randomising table", e );
        }
        MetadataItem metax = new MetadataItem( xnameParam_.stringValue( env ),
                                               xunitParam_.stringValue( env ) );
        MetadataItem metay = new MetadataItem( ynameParam_.stringValue( env ),
                                               yunitParam_.stringValue( env ) );
        TableObservationSetFactory obsFact =
            new TableObservationSetFactory( inTable, metax, metay );
        final InterpolatorFactory interpFact =
            FitUtils.defaultInterpolatorFactory();
        obsFact.setInterpolatorFactory( interpFact );
        obsKeyParam_.configureFactory( env, obsFact );
        final ObservationSet obsSet;
        try {
            obsSet = obsFact.createObservationSet();
        }
        catch ( IOException e ) {
            throw new TaskException( "Can't read observations", e );
        }

        /* Compile expressions for X and Y values. */
        final RandomJELRowReader rowReader = new RandomJELRowReader( inTable );
        String xEx = xParam_.stringValue( env );
        String yEx = yParam_.stringValue( env );
        final CompiledExpression xCex;
        final CompiledExpression yCex;
        if ( "x".equalsIgnoreCase( xEx ) && "y".equalsIgnoreCase( yEx ) ) {
            xCex = null;
            yCex = null;
        }
        else {
            try {
                xCex = Evaluator.compile( xEx, getLibrary( null ),
                                          double.class );
            }
            catch ( CompilationException e ) {
                throw new ParameterValueException( xParam_, e.getMessage(), e );
            }
            try {
                yCex = Evaluator.compile( yEx, getLibrary( rowReader),
                                          double.class );
            }
            catch ( CompilationException e ) {
                throw new ParameterValueException( yParam_, e.getMessage(), e );
            }
        }

        /* Compile expression for redshift values. */
        String zEx = zParam_.stringValue( env );
        final CompiledExpression zCex;
        if ( zEx == null || zEx.trim().length() == 0 ) {
            zCex = null;
        }
        else {
            try {
                zCex = Evaluator.compile( zEx, getLibrary( rowReader ),
                                          double.class );
            }
            catch ( CompilationException e ) {
                throw new ParameterValueException( zParam_, e.getMessage(), e );
            }
        }

        /* Return the executable. */
        final Destination dest = outParam_.destinationValue( env );
        return new Executable() {
            public void execute() throws IOException {
                ObservationSet oset1 =
                    transformObservationSet( obsSet, interpFact, rowReader,
                                             xCex, yCex, zCex );
                new ObservationSetWriter().write( oset1, dest.createStream() );
            }
        };
    }

    /**
     * Transforms the X and Y values and errors in an observation set
     * according to compiled expressions for X and Y.
     * Errors are evaluated using numerical differentiation.
     *
     * @param  obsSet  input observation set
     * @param  interpFact   interpolator factory to generate interpolators
     * @param  rowReader   understands table column names for expression
     *                     evaluations
     * @param  xCex   expression for X
     * @param  yCex   expression for Y
     * @param  zCex   expression for redshift
     * @return   transformed observation set (or possibly the same one)
     */
    private static ObservationSet transformObservationSet(
            ObservationSet obsSet, InterpolatorFactory interpFact,
            RandomJELRowReader rowReader,
            CompiledExpression xCex, CompiledExpression yCex,
            CompiledExpression zCex )
            throws IOException {

        /* If tranformations are null, return the input set. */
        if ( xCex == null && yCex == null && zCex == null ) {
            return obsSet;
        }
        int nobs = obsSet.getObsCount();
        int npoint = obsSet.getPointCount();
        double[] xs = new double[ npoint ];
        double[] xerrs = new double[ npoint ];
        double[][] ys = new double[ npoint ][ nobs ];
        double[][] yerrs = new double[ npoint ][ nobs ];

        /* Prepare JEL evaluation contexts. */
        Obs obs = new Obs( obsSet );

        /* Calculate new X values and errors. */
        if ( xCex == null ) {
            for ( int ip = 0; ip < npoint; ip++ ) {
                xs[ ip ] = obsSet.getX( ip );
                xerrs[ ip ] = obsSet.getXError( ip );
            }
        }
        else {
            Object[] context = new Object[] { obs, };
            for ( int ip = 0; ip < npoint; ip++ ) {
                obs.ipoint_ = ip;
                obs.iobs_ = -1;
                double x;
                double xerr;
                try {
                    obs.xdelta_ = 0.0;
                    obs.ydelta_ = 0.0;
                    xs[ ip ] = xCex.evaluate_double( context );
                    double x0 = obsSet.getX( ip );
                    double xerr0 = obsSet.getXError( ip );
                    double xdelta = 0.001 * xerr0;
                    if ( xdelta > 0 ) {
                        obs.xdelta_ = +xdelta;
                        double fp = xCex.evaluate_double( context );
                        obs.xdelta_ = -xdelta;
                        double fm = xCex.evaluate_double( context );
                        double dfdx = ( fp - fm ) / ( 2 * xdelta );
                        xerrs[ ip ] = Math.abs( dfdx * xerr0 );
                    }
                    else {
                        xerrs[ ip ] = 0;
                    }
                }
                catch ( Throwable e ) {
                    throw (IOException) new IOException( "Evaluation error "
                                                       + e.getMessage() )
                                       .initCause( e );
                }
            }
        }

        /* Calculate new Y values and errors. */
        if ( yCex == null ) {
            for ( int ip = 0; ip < npoint; ip++ ) {
                for ( int iobs = 0; iobs < nobs; iobs++ ) {
                    ys[ ip ][ iobs ] = obsSet.getY( ip, iobs );
                    yerrs[ ip ][ iobs ] = obsSet.getYError( ip, iobs );
                }
            }
        }
        else {
            Object[] rowContext = new Object[] { obs, rowReader, };
            for ( int ip = 0; ip < npoint; ip++ ) {
                obs.ipoint_ = ip;
                for ( int iobs = 0; iobs < nobs; iobs++ ) {
                    obs.iobs_ = iobs;
                    rowReader.setCurrentRow( iobs );
                    try {
                        obs.xdelta_ = 0.0;
                        obs.ydelta_ = 0.0;
                        ys[ ip ][ iobs ] = yCex.evaluate_double( rowContext );
                        double yerr0 = obsSet.getYError( ip, iobs );
                        double ydelta = 0.001 * yerr0;
                        if ( ydelta > 0 ) {
                            obs.ydelta_ = +ydelta;
                            double fp = yCex.evaluate_double( rowContext );
                            obs.ydelta_ = -ydelta;
                            double fm = yCex.evaluate_double( rowContext );
                            double dfdy = ( fp - fm ) / ( 2.0 * ydelta );
                            yerrs[ ip ][ iobs ] = Math.abs( dfdy * yerr0 );
                        }
                        else {
                            yerrs[ ip ][ iobs ] = 0.0;
                        }
                    }
                    catch ( Throwable e ) {
                        throw (IOException) new IOException( "Evaluation error "
                                                           + e.getMessage() )
                                           .initCause( e );
                    }
                }
            }
        }

        /* Get redshifts. */
        double[] xfactors;
        if ( zCex == null ) {
            xfactors = null;
        }
        else {
            Object[] rowContext = new Object[] { obs, rowReader, };
            xfactors = new double[ nobs ];
            for ( int iobs = 0; iobs < nobs; iobs++ ) {
                rowReader.setCurrentRow( iobs );
                double z;
                try {
                    z = zCex.evaluate_double( rowContext );
                }
                catch ( Throwable e ) {
                    throw (IOException) new IOException( "Evaluation error "
                                                       + e.getMessage() )
                                       .initCause( e );
                }
                xfactors[ iobs ] = 1.0 + z;
            }
        }

        /* Provide new interpolators. */
        Interpolator[] interps = new Interpolator[ npoint ];
        for ( int ip = 0; ip < npoint; ip++ ) {
            interps[ ip ] =
                interpFact.getInterpolator( xs[ ip ], xerrs[ ip ] );
        }

        /* Return the transformed observation set. */
        return new ArraysObservationSet( npoint, nobs, xs, xerrs, interps,
                                         ys, yerrs, xfactors,
                                         obsSet.getMetadataX(), 
                                         obsSet.getMetadataY(),
                                         obsSet.getMetadataTable() );
    }

    /**
     * Returns the JEL library for expression evaluations.
     *
     * @return  library
     */
    private static Library getLibrary( JELRowReader rowReader ) {
        Class[] staticLib =
            (Class[]) JELUtils.getStaticClasses().toArray( new Class[ 0 ] );
        Class[] dynamicLib = new Class[] { Obs.class, JELRowReader.class, };
        Class[] dotClasses = new Class[ 0 ];
        return new Library( staticLib, dynamicLib, dotClasses, rowReader,
                            null );
    }

    /**
     * Helper class used for JEL expression evaluation.
     * Constains public methods X (or x) and Y (or y) giving obs X and Y
     * coordinates corresponding to the current values of the 
     * <code>iobs_</code> and <code>ipoint_</code> member variables.
     * <code>xdelta_</code> and <code>ydelta_</code> member variables 
     * adjust these return values.
     */
    public static final class Obs {

        final ObservationSet obsSet_;
        int iobs_ = -1;
        int ipoint_ = -1;
        double xdelta_ = 0.0;
        double ydelta_ = 0.0;

        /**
         * Constructor.
         *
         * @param   obsSet  observation set
         */
        Obs( ObservationSet obsSet ) {
            obsSet_ = obsSet;
        }

        /**
         * Returns X value at current ipoint, iobs plus xdelta.
         *
         * @return  adjusted x
         */
        public double x() {
            return obsSet_.getX( ipoint_ ) + xdelta_;
        }

        /**
         * Returns Y value at current ipoint, iobs plus ydelta.
         *
         * @return  adjusted y
         */
        public double y() {
            return obsSet_.getY( ipoint_, iobs_ ) + ydelta_;
        }

        /**
         * Synonmym for {@link #x}.
         */
        public double X() {
            return x();
        }

        /**
         * Synonym for {@link #y}.
         */
        public double Y() {
            return y();
        }
    }
}
