package fit.test;

import fit.framework.Comparison;
import fit.framework.FitCalculator;
import fit.framework.MetadataTable;
import fit.framework.ObservationSet;
import fit.framework.SampleComparator;
import fit.framework.TheorySet;
import fit.util.ChiSquared;
import fit.util.FitUtils;
import java.util.Arrays;
import java.util.Random;
import junit.framework.TestCase;

/**
 * Performs a full fit of a synthetic observation set against a matched
 * synthetic theory set.
 */
public class FittingTest extends TestCase {

    public FittingTest( String name ) {
        super( name );
    }

    public void testSystem() {

        /* Set up a test data object with low noise.  This means that the
         * fitting should (there's a good chance it will) pick the right 
         * model for each observation. */
        TestData tdata = new TestData( new Random( 1990301L ) );
        tdata.setNoise( 0.01 );

        /* Pick a number of theories which is a multiple of the number of
         * observations, so there is an obvious best one each time. */
        ObservationSet obsSet = tdata.getObservationSet( 20, 17 );
        TheorySet thSet = tdata.getTheorySet( 40, 2000 );

        /* Work out where are certain metadata columns. */
        MetadataTable thMeta = thSet.getMetadataTable();
        MetadataTable obsMeta = obsSet.getMetadataTable();
        int iParamTh = FitUtils.getColumnIndex( thMeta, "DISTORT" );
        int iParamObs = FitUtils.getColumnIndex( obsMeta, "DISTORT" );
        int iYfactObs = FitUtils.getColumnIndex( obsMeta, "YFACT" );
        assertTrue( iParamTh >= 0 );
        assertTrue( iParamObs >= 0 );
        assertTrue( iYfactObs >= 0 );
        assertEquals( -1, FitUtils.getColumnIndex( obsMeta, "Not-here" ) );
        assertEquals( -1, FitUtils.getColumnIndex( thMeta, "Or here." ) );

        /* Iterate over each observation. */
        int nobs = obsSet.getObsCount();
        for ( int iobs = 0; iobs < nobs; iobs++ ) {

            /* Do the fitting. */
            double xfactor = obsSet.getXFactor( iobs ); 
            SampleComparator comparator =
                FitUtils.createComparator( thSet, obsSet, xfactor );
            FitCalculator fitter = new ChiSquared( 1 );
            Comparison[] comparisons =
                comparator.fitData( FitUtils.getYs( obsSet, iobs ),
                                    FitUtils.getYErrors( obsSet, iobs ),
                                    fitter, true );

            /* Sort to get best comparison first. */
            Arrays.sort( comparisons );
            assertTrue( comparisons[ 0 ].getScore()
                        < comparisons[ 1 ].getScore() );

            /* Check that the fitting has picked the right theory out for
             * each observation. */
            Comparison c0 = comparisons[ 0 ];
            int ith = thSet.getIndex( c0.getTheory() );
            double thParam =
                FitUtils.getNumber( thMeta.getValueAt( ith, iParamTh ) );
            double obsParam =
                FitUtils.getNumber( obsMeta.getValueAt( iobs, iParamObs ) );
            assertEquals( obsParam, thParam, 0.0001 );

            /* Check it's recovered close to the correct scaling factor
             * (but not close enough to have cheated). */
            double obsScale =
                FitUtils.getNumber( obsMeta.getValueAt( iobs, iYfactObs ) );
            double thScale = c0.getScale();
            double scaleRatio =
                Math.max( obsScale / thScale, thScale / obsScale );
            assertTrue( Double.toString(scaleRatio),
                        scaleRatio > 1.000001 && scaleRatio < 1.1 );

            /* Check the winning score is near 1, and ones lower down the
             * goodness-of-fit list are worse.  The numbers here are fairly
             * arbitrary (depends on FitCalculator implementation as well
             * as random numbers. */
            double score0 = c0.getScore();
            assertTrue( score0 > 0.1 && score0 < 2.0 );
            assertTrue( comparisons[ 8 ].getScore() > 10.0 );
            assertTrue( comparisons[ comparisons.length-1 ].getScore() > 100 );
        }
    }
}
