package fit.test;

import fit.framework.Theory;
import fit.framework.Interpolator;
import fit.util.ArraysTheory;
import fit.util.LinearInterpolator;
import fit.util.Smoother;
import fit.util.SquareSmoother;
import fit.util.TheoryFactory;
import java.util.Random;
import junit.framework.TestCase;
import uk.ac.starlink.util.DoubleList;

public class InterpolatorTest extends TestCase {

    private final Random random_;

    public InterpolatorTest( String name ) {
        super( name );
        random_ = new Random( 1999L );
    }

    public void testLinearInterpolator() {
        Theory fib = TestUtils.getFibonacciTheory( 10 );
        Interpolator lin = new LinearInterpolator();
        assertTrue( Double.isNaN( lin.getY( fib, -0.1 ) ) );
        assertTrue( Double.isNaN( lin.getY( fib, 10.0 ) ) );
        assertEquals( 0.0, lin.getY( fib, 0.0 ), 0.0 );
        assertEquals( 34.0, lin.getY( fib, 9.0 ), 0.0 );
        assertEquals( 0.5, lin.getY( fib, 0.5 ), 0.0 );
        assertEquals( 1.0, lin.getY( fib, 1.1 ), 0.0 );
        assertEquals( 4.5, lin.getY( fib, 4.75 ), 0.0 );
    }

    public void testSquareSmoother() {
        Theory fib = TestUtils.getFibonacciTheory( 10 );
        Smoother square1 = new SquareSmoother( 1 );
        assertEquals( 2.0, square1.getY( fib, 3.0 ), 0.0 );
        assertTrue( Double.isNaN( square1.getY( fib, 0.4 ) ) );
        assertTrue( Double.isNaN( square1.getY( fib, 8.6 ) ) );
        assertTrue( Double.isNaN( new SquareSmoother( 10 ).getY( fib, 5.0 ) ) );
        assertEquals( 0.5 * ( 8. + 13. ),
                      new SquareSmoother( 2.0 ).getY( fib, 6.5 ), 0.0 );
        assertEquals( 0.5 * ( 8. + 13. ),
                      new SquareSmoother( 0.1 ).getY( fib, 6.5 ), 0.0 );
    }

    public void testLinear() {
        linearExercise( new LinearInterpolator() );
        linearExercise( new SquareSmoother( 1e+1 ) );
        linearExercise( new SquareSmoother( 1e0 ) );
        linearExercise( new SquareSmoother( 1e-1 ) );
        linearExercise( new Smoother() {
            public double[] getBounds( double x ) {
                return new double[] { x - 5, x + 5 };
            }
            public double getWeight( double xMean, double xPoint ) {
                double err = Math.abs( xPoint - xMean );
                return err;
            }
        } );
    }

    private void linearExercise( Interpolator interp ) {
        int np = 100;
        DoubleList xList = new DoubleList();
        for ( double x = -100; x < 100; x += Math.PI / 7 ) {
            xList.add( (double) x );
        }
        double[] xs = xList.toDoubleArray();
        Theory theory1 = new ArraysTheory( "linear-grid", xs, xs );

        TheoryFactory tfact = new TheoryFactory();
        for ( int ip = 0; ip < 1000; ip++ ) {
            double x = ( random_.nextDouble() * 2 - 1 ) * 100.;
            tfact.addPoint( x, x );
        }
        Theory theory2 = tfact.createTheory( "linear-rand" );

        for ( int i = 0; i < 50; i++ ) {
            double x = ( random_.nextDouble() - 0.5 ) * 100;

            /* Tolerances here are a bit ad hoc - depends on exact 
             * characteristics of the interpolators. */
            assertEquals( x, interp.getY( theory1, x ), 0.5 );
            assertEquals( x, interp.getY( theory2, x ), 1.0 );
        }

        double[] bads = new double[] { -1000, Double.NaN, -100.1, 100.1, 1e9, };
        for ( int ibad = 0; ibad < bads.length; ibad++ ) {
            assertTrue( Double.isNaN( interp.getY( theory1, bads[ ibad ] ) ) );
            assertTrue( Double.isNaN( interp.getY( theory2, bads[ ibad ] ) ) );
        }
    }
}
