package fit.test;

import fit.framework.MetadataItem;
import fit.framework.MetadataTable;
import fit.framework.Theory;
import fit.framework.TheorySet;
import fit.util.ArraysTheory;
import fit.util.ArraysTheorySet;
import fit.util.FitUtils;
import fit.util.WrapperTheory;
import junit.framework.TestCase;

public class TheorySetTest extends TestCase {

    public TheorySetTest( String name ) {
        super( name );
    }

    public void testArraysTheorySet() {
        MetadataTable metaTable = null;
        double[] d0 = new double[ 0 ];
        Theory t1, t2, t3, t4;
        Theory[] theories = new Theory[] {
            t1 = new ArraysTheory( "t1", d0, d0 ),
            t2 = new ArraysTheory( "t2", d0, d0 ),
            t3 = new ArraysTheory( "t3", d0, d0 ),
            t4 = new ArraysTheory( "t4", d0, d0 ),
        };
        ArraysTheorySet thSet =
            new ArraysTheorySet( theories, MetadataItem.X, MetadataItem.Y,
                                 metaTable );
        assertEquals( MetadataItem.X, thSet.getMetadataX() );
        assertEquals( MetadataItem.Y, thSet.getMetadataY() );
        checkSet( thSet );
        assertEquals( t1, thSet.getTheory( 0 ) );
        assertEquals( t4, thSet.getTheory( 3 ) );
        try {
            thSet.getTheory( 4 );
            fail();
        }
        catch ( RuntimeException e ) {
        }
        assertEquals( 0, thSet.getIndex( t1 ) );
        assertEquals( 3, thSet.getIndex( t4 ) );

        Theory snafTheory = new ArraysTheory( "t5", d0, d0 );
        assertEquals( -1, thSet.getIndex( snafTheory ) );
        thSet.registerTheory( snafTheory, 2 );
        assertEquals( 2, thSet.getIndex( snafTheory ) );
        thSet.registerTheory( snafTheory, -1 );
        assertEquals( -1, thSet.getIndex( snafTheory ) );

        assertEquals( 3, thSet.getIndex( thSet.getTheory( 3 ) ) );
        assertEquals( 3,
            thSet.getIndex( new WrapperTheory( thSet.getTheory( 3 ) ) ) );
    }

    public void testConcatTheorySet() {
        TheorySet t1 = new DummyTheorySet( "TSA", 5 );
        TheorySet t2 = new DummyTheorySet( "TSB", 15 );
        checkSet( t1 );
        checkSet( t2 );
        TheorySet t12 = FitUtils.concatTheorySets( new TheorySet[] { t1, t2 } );
        checkSet( t12 );
        assertEquals( t1.getTheory( 3 ), t12.getTheory( 3 ) );
        assertEquals( t2.getTheory( 4 ), t12.getTheory( 9 ) );
        assertTrue( ! t2.getTheory( 5 ).equals( t12.getTheory( 9 ) ) );
    }

    private void checkSet( TheorySet theorySet ) {
        int nth = theorySet.getTheoryCount();
        assertTrue( nth >= 0 );
        MetadataTable meta = theorySet.getMetadataTable();
        int ncol = meta == null ? -1 : meta.getColumnCount();
        for ( int ith = 0; ith < nth; ith++ ) {
            Theory th = theorySet.getTheory( ith );
            assertNotNull( th );
            assertEquals( ith, theorySet.getIndex( th ) );
            if ( meta != null ) {
                for ( int icol = 0; icol < ncol; icol++ ) {
                    Object value = meta.getValueAt( ith, icol );
                    if ( value != null ) {
                        assertTrue( meta.getColumnClass( icol )
                                        .isAssignableFrom( value.getClass() ) );
                    }
                }
            }
        }
        assertEquals( -1,
                      theorySet.getIndex( TestUtils.getFibonacciTheory( 1 ) ) );

        if ( meta != null ) {
            assertTrue( FitUtils.isCompatible( meta,
                                               theorySet.getMetadataTable() ) );
        }
    }

    private static class DummyTheorySet implements TheorySet, MetadataTable {
        private final String name_;
        private final int nth_;

        DummyTheorySet( String name, int nth ) {
            name_ = name;
            nth_ = nth;
        }

        public int getTheoryCount() {
            return nth_;
        }

        public Theory getTheory( int ith ) {
            return new DummyTheory( this, ith );
        }

        public int getIndex( Theory theory ) {
            return ( theory instanceof DummyTheory &&
                     ((DummyTheory) theory).set_ == this )
                 ? ((DummyTheory) theory).index_
                 : -1;
        }

        public MetadataItem getMetadataX() {
            return new MetadataItem( "X", null );
        }

        public MetadataItem getMetadataY() {
            return new MetadataItem( "Y", null );
        }

        public MetadataTable getMetadataTable() {
            return this;
        }

        public int getColumnCount() {
            return 2;
        }

        public Class getColumnClass( int icol ) {
            return new Class[] { String.class, Integer.class, }[ icol ];
        }

        public String getColumnName( int icol ) {
            return new String[] { "Name", "Index", }[ icol ];
        }

        public int getRowCount() {
            return nth_;
        }

        public Object getValueAt( int irow, int icol ) {
            return new Object[] {
                name_ + getTheory( irow ).getName(),
                new Integer( irow ),
            }[ icol ];
        }
    }
   
    private static class DummyTheory extends WrapperTheory {
        final TheorySet set_;
        final int index_;
        DummyTheory( TheorySet set, int index ) {
            super( TestUtils.getFibonacciTheory( index * 5 ) );
            set_ = set;
            index_ = index;
        }
        public boolean equals( Object o ) {
            if ( o instanceof DummyTheory ) {
                DummyTheory other = (DummyTheory) o;
                return other.set_.equals( this.set_ )
                    && other.index_ == this.index_;
            }
            return false;
        }
    }
}
