package fit.util;

import java.lang.reflect.Array;
import fit.framework.MetadataTable;

/**
 * MetadataTable implementation based on arrays.
 *
 * @author   Mark Taylor
 * @since    16 Nov 2006
 */
public class ArraysMetadataTable implements MetadataTable {

    private final String[] colNames_;
    private final Object[] colData_;
    private final int nrow_;

    /**
     * Constructor.
     *
     * @param  colNames  column names
     * @param  colData  array of arrays, one for each column; primitive arrays
     *         are permitted
     */
    public ArraysMetadataTable( String[] colNames, Object[] colData ) {
        if ( colNames.length != colData.length ) {
            throw new IllegalArgumentException( "Array length mismatch" );
        }
        nrow_ = Array.getLength( colData[ 0 ] );
        for ( int icol = 0; icol < colNames.length; icol++ ) {
            if ( Array.getLength( colData[ icol ] ) != nrow_ ) {
                throw new IllegalArgumentException( "Array length mismatch" );
            }
        }
        colNames_ = colNames;
        colData_ = colData;
    }

    public String getColumnName( int icol ) {
        return colNames_[ icol ];
    }

    public Class getColumnClass( int icol ) {
        Class clazz = colData_[ icol ].getClass().getComponentType();
        if ( clazz.isPrimitive() ) {
            clazz = Array.get( colData_[ icol ], 0 ).getClass();
        }
        return clazz;
    }

    public int getColumnCount() {
        return colData_.length;
    }

    public int getRowCount() {
        return nrow_;
    }

    public Object getValueAt( int irow, int icol ) {
        return Array.get( colData_[ icol ], irow );
    }
}
